#include "helpers/string_functions.h"

#include <algorithm>
#include <cctype>
#include <sstream>

namespace gtfsplanner {
// trim whitespaces at the front
void ltrim(std::string& s)
{
    s.erase(s.begin(), std::find_if(s.begin(), s.end(), [](char ch) {
                return std::isspace(static_cast<unsigned char>(ch)) == 0;
            }));
}

// trim whitespaces at the end
void rtrim(std::string& s)
{
    s.erase(std::find_if(s.rbegin(), s.rend(),
                         [](char ch) { return std::isspace(static_cast<unsigned char>(ch)) == 0; })
                .base(),
            s.end());
}

void replace_tabs(std::string& s)
{
    std::replace(std::begin(s), std::end(s), '\t', ' ');
}

std::string sanitize_whitespaces(std::string const& input)
{
    std::string result(input);
    ltrim(result);
    rtrim(result);
    replace_tabs(result);
    return result;
}

std::vector<std::string> tokenize_with_quotes(std::string const& input, char delimiter)
{
    std::vector<std::string> result;

    bool inside_quotation = false;
    std::string token;
    for (auto i = 0U; i < input.size(); i++)
    {
        if (input[i] == '"' && ((i + 1) < input.size()) && input[i + 1] == '"')
        {
            // begin of a double quotation mark, just ignore this char
            continue;
        }

        if (input[i] == '"' && (i == 0 || input[i - 1] != '"')) // previous char is not "
        {
            inside_quotation = !inside_quotation;
        }
        else if (input[i] == delimiter && !inside_quotation)
        {
            // token ends here
            result.push_back(sanitize_whitespaces(token));
            token.clear();
        }
        else
        {
            token += input[i];
        }
    }
    // add the last token when the input ends
    result.push_back(sanitize_whitespaces(token));

    return result;
}

std::vector<std::string> tokenize(std::string const& input, char delimiter)
{
    std::vector<std::string> tokens;
    std::stringstream ss(input);
    size_t i = 0;
    std::string token;
    while (std::getline(ss, token, delimiter))
    {
        tokens.push_back(token);
    }
    return tokens;
}

size_t fuzzy_string_compare(std::string const& a, std::string const& b)
{
    size_t m = a.size();
    size_t n = b.size();
    std::vector<std::vector<int>> matrix;
    matrix.resize(m + 1);
    for (auto& entry : matrix)
    {
        entry.resize(n + 1);
    }
    for (int i = 0; i <= m; i++)
    {
        matrix[i][0] = 0;
    }
    for (int j = 0; j <= n; j++)
    {
        matrix[0][j] = 0;
    }
    for (int i = 1; i <= m; i++)
    {
        for (int j = 1; j <= n; j++)
        {
            if (a[i - 1] == b[j - 1])
            {
                matrix[i][j] = matrix[i - 1][j - 1] + 1;
            }
            else
            {
                matrix[i][j] = std::max(matrix[i][j - 1], matrix[i - 1][j]);
            }
        }
    }
    return matrix[m][n];
}

std::string to_upper(std::string const& input)
{
    std::string result(input);
    // first replace utf8 special chars
    result = std::regex_replace(result, std::regex("\xC3\xA4"), "\xC3\x84"); //  => 
    result = std::regex_replace(result, std::regex("\xC3\xB6"), "\xC3\x96"); //  => 
    result = std::regex_replace(result, std::regex("\xC3\xBC"), "\xC3\x9C"); //  => 
    for (auto& c : result)
    {
        if (c >= 'a' && c <= 'z')
        {
            c = c - 'a' + 'A';
        }
    }
    return result;
}

size_t length_utf8(std::string const& s)
{
    size_t count = 0;
    for (auto const c : s)
    {
        // bitmask tells whether this is an actual char or something in a multibyte encoding
        if ((c & 0xC0) != 0x80)
        {
            count++;
        }
    }
    return count;
}

} // namespace gtfsplanner
