#include "helpers/progress_writer.h"

#include "helpers/console.h"

#include <sstream>

namespace gtfsplanner {
Progress_writer::Progress_writer(std::string const& title, std::streamoff total)
    : m_total(total), m_current(0), m_current_percentage(0)
{
    m_start = std::chrono::steady_clock::now();
    Console::write(title + " ");
}
Progress_writer::~Progress_writer()
{
    std::ostringstream ss;
    ss << "done (" << get_elapsed_time() << ")" << std::endl;
    Console::write(ss.str());
}

std::string format_time(int64_t time_in_sec)
{
    auto hours = time_in_sec / 3600;
    auto minutes = (time_in_sec - (hours * 3600)) / 60;
    auto seconds = time_in_sec % 60;

    std::string result;
    char unit = 0;
    if (hours > 0)
    {
        result += std::to_string(hours);
        result += ':';
        unit = 'h';
    }
    if (minutes > 0)
    {
        if (!result.empty() && minutes < 10)
        {
            result += '0';
        }
        result += std::to_string(minutes);
        result += ':';
        if (unit == 0)
        {
            unit = 'm';
        }
    }
    if (!result.empty() && seconds < 10)
    {
        result += '0';
    }
    result += std::to_string(seconds);
    if (unit == 0)
    {
        unit = 's';
    }
    result += unit;
    return result;
}

std::string Progress_writer::get_elapsed_time()
{
    auto now = std::chrono::steady_clock::now();
    auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(now - m_start).count();
    return format_time(elapsed);
}

void Progress_writer::update(std::streamoff current)
{
    m_current += current;
    std::streamoff percentage = m_current * 50 / m_total;
    if (percentage > m_current_percentage)
    {
        auto now = std::chrono::steady_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(now - m_start).count();
        if (elapsed > 0)
        {
            if (m_current_percentage == 0)
            {
                Console::write("\n");
            }
            // it took elapsed seconds to get to the current percentage, now extrapolate
            // how many seconds are left
            auto remaining = (elapsed * (50 - percentage) / percentage);
            auto time_remaining = format_time(remaining);

            for (auto i = m_current_percentage; i < percentage; i++)
            {
                if ((i % 10) == 0)
                {
                    Console::write(std::to_string(2 * i) + "%");
                }
                else if ((i % 5) == 0)
                {
                    Console::write("|");
                }
                else
                {
                    Console::write(".");
                }
            }
            Console::write(" " + time_remaining);
            for (auto i = 0U; i <= time_remaining.size(); i++)
            {
                Console::write("\b");
            }
            m_current_percentage = percentage;
        }
        Console::flush();
    }
}

} // namespace gtfsplanner
