#include "helpers/map_writer.h"

#include "helpers/heatmap.h"

#include <algorithm>
#include <fstream>
#include <functional>
#include <iomanip>
#include <sstream>

namespace gtfsplanner {
template <typename T>
struct Label
{
    std::string name;
    double lon = 0.0;
    double lat = 0.0;
    Heatmap::Color color {0, 0, 0};
    double scaled_value = 0.0;
    T total_value;
};

template <typename T>
void write(gtfs::Dataset const& dataset,
           std::string const& filename,
           std::string const& mode,
           std::string const& heatmode,
           std::unordered_map<std::string, T> const& count,
           std::function<double(T const&, T const&, T const&)> f)
{
    // calculate the colors for the heatmap - take min and max values and then it's a linear interpolation with
    auto minmax = std::minmax_element(
        count.begin(), count.end(),
        [](std::pair<std::string, T> const& lhs, std::pair<std::string, T> const& rhs) {
            return lhs.second < rhs.second;
        });

    Heatmap heatmap;
    std::unordered_map<std::string, Heatmap::Gradient> mapping = {
        {"rainbow", Heatmap::Gradient::Rainbow},
        {"monochrome", Heatmap::Gradient::Monochrome},
        {"viridis", Heatmap::Gradient::Viridis},
        {"inferno", Heatmap::Gradient::Inferno},
        {"wistia", Heatmap::Gradient::Wistia}};
    auto iter = mapping.find(heatmode);
    if (iter != mapping.end())
    {
        heatmap.set_gradient(iter->second);
    }

    // create the actual label entry, but sort them by alphabet
    std::vector<Label<T>> labels;
    for (auto const& entry : count)
    {
        // figure out the station and create a placemark for it
        auto stop_iter = dataset.stop_id_to_idx.find(entry.first);
        check_iter(stop_iter, dataset.stop_id_to_idx);
        auto const& stop = dataset.stops[stop_iter->second];

        // map value to range 0.0 - 1.0
        auto value = f(entry.second, minmax.first->second, minmax.second->second);
        auto color = heatmap.get_color(value);
        labels.push_back({stop.name, stop.lon, stop.lat, color, value, entry.second});
    }
    std::sort(labels.begin(), labels.end(),
              [](Label<T> const& lhs, Label<T> const& rhs) { return lhs.name < rhs.name; });

    std::ofstream out(filename);
    if (!out.is_open())
    {
        throw File_error("Could not open " + filename + " for writing.");
    }


    if (mode == "kml")
    {
        // write a kml file
        out << R"(<?xml version="1.0" encoding="UTF-8"?>)" << std::endl;
        out << R"(<kml xmlns="http://earth.google.com/kml/2.0">)" << std::endl;
        out << R"(<Document>)" << std::endl;
        // define the style - marker that can be colored, name will only be displayed when highlighting it on the map
        // to not clutter the map with labels
        out << R"(<Style id="defaultIcon">
	<LabelStyle>
		<scale>0</scale>
	</LabelStyle>
	<IconStyle>
		<Icon><href>http://maps.google.com/mapfiles/kml/shapes/placemark_circle.png</href></Icon>
		<colorMode>normal</colorMode>
	</IconStyle>
    <BalloonStyle>
      <text>$[description]</text>
    </BalloonStyle>
</Style>
<Style id="hoverIcon">
	<IconStyle>
		<Icon><href>http://maps.google.com/mapfiles/kml/shapes/placemark_circle.png</href></Icon>
		<colorMode>normal</colorMode>
	</IconStyle>
    <BalloonStyle>
      <text>$[description]</text>
    </BalloonStyle>
</Style>
<StyleMap id="defaultStyle">
	<Pair>
		<key>normal</key>
		<styleUrl>#defaultIcon</styleUrl>
	</Pair>
	<Pair>
		<key>highlight</key>
		<styleUrl>#hoverIcon</styleUrl>
	</Pair>
</StyleMap>)"
            << std::endl;

        for (auto const& label : labels)
        {
            std::stringstream ss;
            ss << "ff";
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.b);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.g);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.r);

            out << "<Placemark><styleUrl>#defaultStyle</styleUrl>" << std::endl;
            out << "<Style><IconStyle><scale>" << 1.0 + label.scaled_value << "</scale><color>"
                << ss.str() << "</color></IconStyle></Style>" << std::endl;
            out << "<name>" << label.name << "</name>" << std::endl;
            out << "<Point>" << std::endl;
            out << "<coordinates>" << label.lon << "," << label.lat << ", 0."
                << "</coordinates>" << std::endl;
            out << "</Point>" << std::endl;
            out << "<description>" << label.name << ": " << label.total_value << "</description>"
                << std::endl;
            out << "</Placemark>" << std::endl;
        }

        out << R"(</Document></kml>)";
    }
    else
    {
        // geojson, since that is the only other supported format
        out << R"({ "type":"FeatureCollection","features":[)" << std::endl;
        for (auto i = 0U; i < labels.size(); i++)
        {
            auto const& label = labels[i];
            std::stringstream ss;
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.r);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.g);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.b);
            if (i > 0)
            {
                out << ",";
            }
            out << R"({"type":"Feature","properties":{)" << std::endl;
            out << R"("name":")" << label.name << R"(",)";
            out << R"("_umap_options":{"color": "#)" << ss.str() << R"("})";
            out << R"(}, "geometry": {"type":"Point","coordinates": [)" << std::endl;
            out << label.lon << "," << label.lat;
            out << R"(] })" << std::endl;
            out << "}" << std::endl;
        }
        out << R"(] })" << std::endl;
    }
}

void write_heatmap(gtfs::Dataset const& dataset,
                   std::string const& filename,
                   std::string const& mode,
                   std::string const& heatmode,
                   std::unordered_map<std::string, size_t> const& count)
{
    write<size_t>(dataset, filename, mode, heatmode, count,
                  [](size_t const& value, size_t const& minimum, size_t const& maximum) {
                      return static_cast<double>(value - minimum)
                             / static_cast<double>(maximum - minimum);
                  });
}

void write_heatmap(gtfs::Dataset const& dataset,
                   std::string const& filename,
                   std::string const& mode,
                   std::string const& heatmode,
                   std::unordered_map<std::string, Time> const& count)
{
    write<Time>(
        dataset, filename, mode, heatmode, count,
        [](Time const& value, Time const& minimum, Time const& maximum) {
            auto value_s = value.hour * 3600 + value.minute * 60 + value.second;
            auto min_s = minimum.hour * 3600 + minimum.minute * 60 + minimum.second;
            auto max_s = maximum.hour * 3600 + maximum.minute * 60 + maximum.second;
            // use 1.0 - because we need to inverse the heat map - the less time the better
            return 1.0 - static_cast<double>(value_s - min_s) / static_cast<double>(max_s - min_s);
        });
}
} // namespace gtfsplanner
