#include "helpers/console.h"

#include "helpers/string_functions.h"

#include <algorithm>
#include <iostream>
#include <sstream>
#include <string>
#include <unordered_map>

namespace gtfsplanner {
Console& Console::get()
{
    static Console instance;
    return instance;
}

std::string convert_from_utf8(std::unordered_map<std::string, char> const& replacements,
                              std::string const& input)
{
    std::string output(input);
    for (auto const& replace : replacements)
    {
        std::string tmp;
        tmp += replace.second;
        output = std::regex_replace(output, std::regex(replace.first), tmp);
    }
    return output;
}

std::string convert_utf8_tocp850(std::string const& input)
{
    // there's probably more elegant ways to do this, but a simple lookup table for the relevant characters should be helpful enough
    std::unordered_map<std::string, char> replacements = {
        {"\xC3\xA4", 0x84}, //
        {"\xC3\xB6", 0x94}, //
        {"\xC3\xBC", 0x81}, //
        {"\xC3\x9F", 0xE1}, //
        {"\xC3\x84", 0x8E}, //
        {"\xC3\x96", 0x99}, //
        {"\xC3\x9C", 0x9A}, //
        {"\xC3\xA8", 0x8A}, //
        {"\xC3\xA9", 0x82}, //
        {"\xC5\xBD", 'Z'}   //
    };
    return convert_from_utf8(replacements, input);
}

void Console::write(std::string const& out)
{
    if (!out.empty())
    {
        Console::get().write_impl(out);
    }
}

void Console::write_impl(std::string const& out)
{
    // keep track of cursor and insert line breaks in appropriate places
    size_t cursor = 0;
    bool is_first = true; // valid for first word on line, no space needed before it
    for (auto i = 0U; i < m_indent; i++)
    {
        std::cout << " ";
        cursor++;
    }
    std::vector<std::string> words = tokenize(out, ' ');
    for (auto const& word : words)
    {
        // replace non-breaking spaces
        auto spaced_word = std::regex_replace(word, std::regex(nbsp), " ");
// Linux has working UTF8 on the terminal
#if defined _WIN32
        spaced_word = convert_utf8_tocp850(spaced_word);
#endif
        auto word_length = spaced_word.size();
        if (cursor + word_length > 78)
        {
            std::cout << "\n";
            cursor = 0;
            for (auto i = 0U; i < m_indent; i++)
            {
                std::cout << " ";
                cursor++;
            }
            is_first = true;
        }
        if (!is_first)
        {
            std::cout << " ";
            cursor++;
        }
        std::cout << spaced_word;
        is_first = false;
        cursor += word_length;
    }
    if (out.back() == ' ')
    {
        std::cout << " ";
    }
}

size_t count_special_chars(std::string const& chars, std::string const& input)
{
    return std::count_if(input.begin(), input.end(),
                         [&chars](char c) { return chars.find(c) != std::string::npos; });
}

bool is_cp850(std::string const& input)
{
    //                                      
    std::string needle("\x84\x94\x81\xE1\x8E\x99\x9A");
    return count_special_chars(needle, input) > 0;
}

bool is_cp1252(std::string const& input)
{
    //                                      
    std::string needle("\xE4\xF6\xFC\xDF\xC4\xD6\xDC");
    return count_special_chars(needle, input) > 0;
}

std::string convert_to_utf8(std::unordered_map<char, std::string> const& replacements,
                            std::string const& input)
{
    std::string output;
    for (auto const& c : input)
    {
        auto iter = replacements.find(c);
        if (iter == replacements.end())
        {
            if (static_cast<unsigned char>(c) > 127)
            {
                std::stringstream ss;
                ss << "Found invalid character with hex code " << static_cast<unsigned>(c)
                   << ", displayed as " << c
                   << ". This will lead to wrong results on whatever operation you're trying to "
                      "accomplish. Get on the nerves of the developer to add support for this "
                      "special character!\n";
                Console::write(ss.str());
            }
            output.push_back(c);
        }
        else
        {
            output.append(iter->second);
        }
    }
    return output;
}

std::string convert_cp850_to_utf8(std::string const& input)
{
    // there's probably more elegant ways to do this, but a simple lookup table for the relevant characters should be helpful enough
    std::unordered_map<char, std::string> replacements = {
        {0x84, "\xC3\xA4"}, {0x94, "\xC3\xB6"}, {0x81, "\xC3\xBC"},
        {0xE1, "\xC3\x9F"}, {0xBE, "\xC3\x84"}, {0x99, "\xC3\x96"},
        {0x9A, "\xC3\x9C"}, {0x8A, "\xC3\xA8"}, {0x82, "\xC3\xA9"}};
    return convert_to_utf8(replacements, input);
}

std::string convert_cp1252_to_utf8(std::string const& input)
{
    // there's probably more elegant ways to do this, but a simple lookup table for the relevant characters should be helpful enough
    std::unordered_map<char, std::string> replacements = {
        {0xE4, "\xC3\xA4"}, {0xF6, "\xC3\xB6"}, {0xFC, "\xC3\xBC"}, {0xDF, "\xC3\x9F"},
        {0xC4, "\xC3\x84"}, {0xD6, "\xC3\x96"}, {0xDC, "\xC3\x9C"}};
    return convert_to_utf8(replacements, input);
}

std::string Console::read()
{
    return Console::get().read_impl();
}

std::string Console::read_impl()
{
    std::string input;
    std::getline(std::cin, input);

    // heuristically determine encoding by checking expected special chars 
    // This is required since Windows console used whatever the heck they used
    // in ancient times. This code works on heuristics and might be broken in
    // subtle cases - only German umlauts are checked, not the full codepage.
    if (is_cp850(input) && !is_cp1252(input))
    {
        input = convert_cp850_to_utf8(input);
    }
    else if (is_cp1252(input))
    {
        input = convert_cp1252_to_utf8(input);
    }
    else
    {
        // nothing to do if neither cp850 nor cp1252 detected
    }

    return input;
}

bool Console::eof()
{
    return Console::get().eof_impl();
}

bool Console::eof_impl()
{
    return std::cin.eof();
}

void Console::flush()
{
    Console::get().flush_impl();
}

void Console::flush_impl()
{
    std::cout.flush();
}
} // namespace gtfsplanner
