#include "commands/exit.h"
#include "commands/parser.h"
#include "gtfs_types.h"
#include "helpers/console.h"

#include <iostream>
#include <stdexcept>
#include <string>
#include <vector>

constexpr const char* version = "1.0.1";

namespace gtfsplanner {
gtfs::Dataset dataset; // the dataset on which to work

void run(std::vector<std::string> const& args)
{
    std::vector<std::unique_ptr<Command>> commands = parse(args);

    // run the main loop
    for (auto const& cmd : commands)
    {
        cmd->sanitize();
        cmd->execute(dataset);
    }
    commands.clear();

    // now run the internal command console
    while (!exit_called() && !Console::eof())
    {
        try
        {

            Console::write("> ");
            std::string input = Console::read();
            auto command = parse_command(input);
            if (command)
            {
                command->sanitize();
                command->execute(dataset);
            }
        }
        catch (std::exception const& e)
        {
            Console::write(std::string("Error: ") + e.what() + "\n");
        }
    }
}
} // namespace gtfsplanner

int main(int argc, char** argv)
{
    try
    {
        // collect the commandline parameters, ignore the first one (it's the name of the program
        std::vector<std::string> args;
        for (auto i = 1U; i < argc; i++)
        {
            args.emplace_back(argv[i]); // NOLINT - cannot avoid pointer arithmetics here
        }

        gtfsplanner::Console::write(std::string("gtfsplanner ") + version + std::string("\n"));
        gtfsplanner::Console::write(
            "Type \"help\" to get a list of functions or \"exit\" to stop the program.\n");

        gtfsplanner::run(args);
        return EXIT_SUCCESS;
    }
    catch (std::exception const& e)
    {
        std::cerr << "Error: " << e.what() << std::endl;
        return EXIT_FAILURE;
    }
}
