#include "gtfs_helpers.h"

#include "helpers/console.h"
#include "helpers/string_functions.h"

#include <algorithm>
#include <array>
#include <regex>
#include <sstream>

namespace gtfsplanner {
std::string build_route_name(gtfs::Dataset const& dataset, gtfs::Trip const& trip)
{
    auto route_id = trip.route_id;
    auto route_iter = dataset.route_id_to_idx.find(route_id);
    check_iter(route_iter, dataset.route_id_to_idx);
    auto const& route = dataset.routes[route_iter->second];
    return route.short_name + " " + trip.id;
}

std::vector<Trip_stop> build_trip_stops(gtfs::Dataset const& dataset, gtfs::Trip const& trip)
{
    std::vector<Trip_stop> result;
    for (auto const& stop_time_idx : trip.stop_time_indices)
    {
        auto const& stop_time = dataset.stop_times[stop_time_idx];
        auto stop_iter = dataset.stop_id_to_idx.find(stop_time.stop_id);
        check_iter(stop_iter, dataset.stop_id_to_idx);
        auto const& stop = dataset.stops[stop_iter->second];
        result.push_back({stop.id, stop.name, stop_time.arrival_time, stop_time.departure_time});
    }
    return result;
}

Trip_data build_trip_data(gtfs::Dataset const& dataset, std::string const& trip_id)
{
    auto trip_iter = dataset.trip_id_to_idx.find(trip_id);
    check_iter(trip_iter, dataset.trip_id_to_idx);
    auto const& trip = dataset.trips[trip_iter->second];

    Trip_data result;
    result.service_calendar.start_date = {0, 0, 0};
    result.service_calendar.end_date = {0, 0, 0};
    result.trip_type_name = build_route_name(dataset, trip); // e.g. ICE 0815
    result.trip_id = trip.id;
    result.stops = build_trip_stops(dataset, trip);
    auto service_id = trip.service_id;
    auto service_iterator = std::find_if(dataset.calendars.begin(), dataset.calendars.end(),
                                         [&service_id](gtfs::Calendar const& calendar) {
                                             return calendar.service_id == service_id;
                                         });
    if (service_iterator != dataset.calendars.end())
    {
        result.service_calendar = *service_iterator;
    }
    for (auto const& date : dataset.calendar_dates)
    {
        if (date.service_id == service_id)
        {
            result.service_exceptions.push_back(date);
        }
    }
    return result;
}

void print(Trip_data const& trip, Trip_mode mode)
{
    Console::write(trip.trip_type_name + " from " + trip.stops.front().name + " to "
                   + trip.stops.back().name + "\n");
    print_trip_stops(trip, mode);
    print_trip_service_days(trip);
    Console::write("\n");
}

std::string create_service_calendar_string(gtfs::Calendar const& c)
{
    // check for common patterns:
    std::stringstream ss;
    std::vector<std::pair<bool, std::string>> days {
        {c.monday, "Mo"}, {c.tuesday, "Tu"},  {c.wednesday, "We"}, {c.thursday, "Th"},
        {c.friday, "Fr"}, {c.saturday, "Sa"}, {c.sunday, "Su"}};
    if (c.monday && c.tuesday && c.wednesday && c.thursday && c.friday && c.saturday && c.sunday)
    {
        ss << "Daily";
    }
    else if (c.monday && c.tuesday && c.wednesday && c.thursday && c.friday && !c.saturday
             && !c.sunday)
    {
        ss << "Mo-Fr";
    }
    else if (std::count_if(days.begin(), days.end(),
                           [](std::pair<bool, std::string> const& day) { return day.first; })
             == 0)
    {
        return ""; // does not travel at all in a regular interval - only based on exceptions
    }
    else if (std::count_if(days.begin(), days.end(),
                           [](std::pair<bool, std::string> const& day) { return day.first; })
             == 6)
    {
        ss << "Daily except ";
        auto iter =
            std::find_if(days.begin(), days.end(),
                         [](std::pair<bool, std::string> const& day) { return !day.first; });
        ss << iter->second;
    }
    else
    {
        // list every single day
        ss << "Every ";
        bool comma_needed = false;
        for (auto const& day : days)
        {
            if (day.first)
            {
                if (comma_needed)
                {
                    ss << ", ";
                }
                ss << day.second;
                comma_needed = true;
            }
        }
    }

    ss << ", from " << c.start_date << " to " << c.end_date;
    return ss.str();
}

std::string add_dates(std::vector<gtfs::Calendar_date> const& service_exceptions,
                      std::string const& prefix,
                      gtfs::Service_exception type)
{
    std::stringstream ss;
    ss << prefix;
    bool comma_needed = false;
    for (auto const& date : service_exceptions)
    {
        if (date.exception_type == type)
        {
            if (comma_needed)
            {
                ss << ", ";
            }
            ss << date.date;
            comma_needed = true;
        }
    }
    return ss.str();
}

std::string
create_service_exception_string(std::vector<gtfs::Calendar_date> const& service_exceptions)
{
    bool has_added = std::find_if(service_exceptions.begin(), service_exceptions.end(),
                                  [](gtfs::Calendar_date const& date) {
                                      return date.exception_type == gtfs::Service_exception::ADDED;
                                  })
                     != service_exceptions.end();
    bool has_removed =
        std::find_if(service_exceptions.begin(), service_exceptions.end(),
                     [](gtfs::Calendar_date const& date) {
                         return date.exception_type == gtfs::Service_exception::REMOVED;
                     })
        != service_exceptions.end();
    // collect all added dates, then all excluded dates
    std::stringstream ss;
    if (has_added)
    {
        ss << add_dates(service_exceptions, "Travels on ", gtfs::Service_exception::ADDED);
    }
    if (has_added && has_removed)
    {
        ss << std::endl;
    }
    if (has_removed)
    {
        ss << add_dates(service_exceptions, "Not on ", gtfs::Service_exception::REMOVED);
    }
    return ss.str();
}

void print_trip_service_days(Trip_data const& trip)
{
    Console::write(create_service_calendar_string(trip.service_calendar));
    if (!trip.service_exceptions.empty())
    {
        Console::write(create_service_exception_string(trip.service_exceptions));
    }
}

void print_trip_stops(Trip_data const& trip, Trip_mode mode)
{
    std::stringstream ss;
    for (auto i = 0U; i < trip.stops.size(); i++)
    {
        auto const& stop = trip.stops[i];
        if (i > 0)
        {
            ss << ", ";
        }
        // replace spaces in station name and between station and departure time with non-breaking space
        // this controls line breaks in the console, the console write function will turn it back into regular spaces
        // but maintain the non-breaking rule
        auto name = std::regex_replace(stop.name, std::regex(" "), nbsp);
        ss << name << nbsp;
        if (mode == Trip_mode::ARRIVAL)
        {
            ss << stop.departure;
        }
        else
        {
            ss << stop.arrival;
        }
    }
    ss << "\n";
    Console::write(ss.str());
}

std::vector<gtfs::Stop>::const_iterator find_station(gtfs::Dataset const& dataset,
                                                     std::string const& station)
{
    // first, look for the station name in the stations
    auto iter = std::find_if(dataset.stops.begin(), dataset.stops.end(),
                             [&station](gtfs::Stop const& stop) { return stop.name == station; });
    if (iter == dataset.stops.end())
    {
        // no fitting station found, show similarly named stations
        std::vector<std::pair<std::string, size_t>> distances;
        distances.reserve(dataset.stops.size());
        for (auto const& stop : dataset.stops)
        {
            auto distance = fuzzy_string_compare(stop.name, station);
            distances.emplace_back(std::make_pair(stop.name, distance));
        }
        std::sort(
            distances.begin(), distances.end(),
            [](std::pair<std::string, size_t> const& lhs,
               std::pair<std::string, size_t> const& rhs) { return lhs.second > rhs.second; });
        Console::write("Could not find exact match for " + station + ", found similar names:\n");
        for (auto i = 0U; i < 10 && i < distances.size(); i++)
        {
            if (distances[i].second >= station.length() - 2)
            {
                Console::write(distances[i].first + "\n");
            }
        }
    }
    return iter;
}

bool is_active_on_date(gtfs::Calendar const& calendar,
                       std::vector<gtfs::Calendar_date> const& exceptions,
                       Date const& date)
{
    bool result = false;
    // service may be relevant
    size_t dayofweek = get_day_of_week(date);
    std::array<bool, 7> active = {calendar.sunday,    calendar.monday,   calendar.tuesday,
                                  calendar.wednesday, calendar.thursday, calendar.friday,
                                  calendar.saturday};

    result = active[dayofweek];
    for (auto const& exception : exceptions)
    {
        if (exception.date == date)
        {
            return (exception.exception_type == gtfs::Service_exception::ADDED);
        }
    }
    return result;
}

bool is_active_on_date(gtfs::Dataset const& dataset,
                       std::string const& service_id,
                       Date const& date)
{
    bool result = false;
    for (auto const& calendar : dataset.calendars)
    {
        if (calendar.service_id == service_id && calendar.start_date <= date
            && calendar.end_date >= date)
        {
            // service may be relevant
            size_t dayofweek = get_day_of_week(date);
            std::array<bool, 7> active = {calendar.sunday,    calendar.monday,   calendar.tuesday,
                                          calendar.wednesday, calendar.thursday, calendar.friday,
                                          calendar.saturday};
            result = active[dayofweek];
            break;
        }
    }

    for (auto const& calendar_date : dataset.calendar_dates)
    {
        if (calendar_date.service_id == service_id && calendar_date.date == date)
        {
            return (calendar_date.exception_type == gtfs::Service_exception::ADDED);
        }
    }
    return result;
}

bool fits_time_window(Time stop_time, Time const& start, Time const& end)
{
    // due to the way the data is stored, times can mark departures on the next day or even further for long trips
    // prominent example: Transsibirian railway, or simply overnight connections
    while (stop_time.hour > 24)
    {
        stop_time.hour -= 24;
    }
    return start <= stop_time && stop_time <= end;
}

std::vector<Trip_data> get_departures(gtfs::Dataset const& dataset,
                                      std::string const& station_id,
                                      Date const& date,
                                      Time const& start,
                                      Time const& end)
{
    std::vector<Trip_data> result;
    for (auto const& stop_time : dataset.stop_times)
    {
        // iterate over all stop times to figure out which ones are inside the given filter criteria
        // first, filter by station_id (obvious)
        if (stop_time.stop_id != station_id
            || !fits_time_window(stop_time.departure_time, start, end))
        {
            continue;
        }
        // found a trip that might fit, now check if the stop_time is actually a departure, or rather
        // the last stop on the trip
        auto trip_id = stop_time.trip_id;
        auto trip_iter = dataset.trip_id_to_idx.find(trip_id);
        check_iter(trip_iter, dataset.trip_id_to_idx);
        auto const& trip = dataset.trips[trip_iter->second];
        if (stop_time.stop_sequence == trip.stop_time_indices.size() - 1)
        {
            // train does not have any further stops after this one, so it's not a departure
            continue;
        }
        // check the date
        if (date != Date {0, 0, 0} && !is_active_on_date(dataset, trip.service_id, date))
        {
            // train does not travel on the given date
            continue;
        }

        Trip_data departure = build_trip_data(dataset, trip_id);
        // remove all stops that are before the current station
        departure.stops.erase(departure.stops.begin(),
                              departure.stops.begin() + stop_time.stop_sequence);
        result.push_back(departure);
    }
    // collected all departures, now sort them and show them
    std::sort(result.begin(), result.end(), [](Trip_data const& lhs, Trip_data const& rhs) {
        auto time_l = lhs.stops[0].arrival;
        while (time_l.hour >= 24)
        {
            time_l.hour -= 24;
        }
        auto time_r = rhs.stops[0].arrival;
        while (time_r.hour >= 24)
        {
            time_r.hour -= 24;
        }
        return time_l < time_r;
    });
    return result;
}

std::vector<Trip_data> get_arrivals(gtfs::Dataset const& dataset,
                                    std::string const& station_id,
                                    Date const& date,
                                    Time const& start,
                                    Time const& end)
{
    std::vector<Trip_data> result;
    for (auto const& stop_time : dataset.stop_times)
    {
        // iterate over all stop times to figure out which ones are inside the given filter criteria
        // first, filter by station_id (obvious)
        if (stop_time.stop_id != station_id
            || !fits_time_window(stop_time.arrival_time, start, end))
        {
            continue;
        }
        // found a trip that might fit, now check if the stop_time is actually an arrival, or rather
        // the first stop on the trip
        auto trip_id = stop_time.trip_id;
        auto trip_iter = dataset.trip_id_to_idx.find(trip_id);
        check_iter(trip_iter, dataset.trip_id_to_idx);
        auto const& trip = dataset.trips[trip_iter->second];
        if (stop_time.stop_sequence == 0)
        {
            // train does not have any further stops before this one, so it's not an arrival
            continue;
        }
        // check the date
        if (date != Date {0, 0, 0} && !is_active_on_date(dataset, trip.service_id, date))
        {
            // train does not travel on the given date
            continue;
        }

        Trip_data arrival = build_trip_data(dataset, trip_id);
        // remove all stops that are after the current station
        arrival.stops.erase(arrival.stops.begin() + stop_time.stop_sequence + 1,
                            arrival.stops.end());
        result.push_back(arrival);
    }
    // collected all departures, now sort them and show them
    std::sort(result.begin(), result.end(), [](Trip_data const& lhs, Trip_data const& rhs) {
        auto time_l = lhs.stops[lhs.stops.size() - 1].arrival;
        while (time_l.hour >= 24)
        {
            time_l.hour -= 24;
        }
        auto time_r = rhs.stops[rhs.stops.size() - 1].arrival;
        while (time_r.hour >= 24)
        {
            time_r.hour -= 24;
        }
        return time_l < time_r;
    });
    return result;
}
} // namespace gtfsplanner
