#include "commands/trip.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/string_functions.h"

namespace gtfsplanner {
Trip_cmd::Trip_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}


void Trip_cmd::sanitize()
{
    auto const& params = get_parameters();
    if (params.empty())
    {
        error("Missing parameter for trip command. Name has to be given");
    }
    if (!params.empty())
    {
        m_trip_name = params[0];
    }
    check_regex(m_trip_name);
}


void Trip_cmd::execute(gtfs::Dataset& dataset)
{
    std::regex filter(m_trip_name);

    std::vector<Trip_data> result;
    for (auto const& trip : dataset.trips)
    {
        auto name = build_route_name(dataset, trip);
        if (std::regex_match(name, filter))
        {
            result.push_back(build_trip_data(dataset, trip.id));
        }
    }
    std::sort(result.begin(), result.end(), [](Trip_data const& lhs, Trip_data const& rhs) {
        return lhs.trip_type_name < rhs.trip_type_name;
    });

    size_t longest_name = 0;
    for (auto const& data : result)
    {
        for (auto const& stop : data.stops)
        {
            longest_name = std::max(longest_name, length_utf8(stop.name) + 1);
        }
    }
    for (auto const& data : result)
    {
        Console::write(data.trip_type_name + " from " + data.stops.front().name + " to "
                       + data.stops.back().name + "\n");

        // prepare for print output

        Console::get().set_indent(longest_name);
        Console::write(std::string("arrival") + nbsp + nbsp + "departure");
        Console::get().set_indent(0);
        Console::write("\n");
        for (auto i = 0U; i < data.stops.size(); i++)
        {
            auto const& entry = data.stops[i];
            Console::write(entry.name);
            for (auto j = 0U; j < longest_name - length_utf8(entry.name); j++)
            {
                Console::write(nbsp);
            }
            if (i == 0)
            {
                for (auto k = 0U; k < 9; k++)
                {
                    Console::write(nbsp);
                }
                Console::write(to_string(entry.departure) + "\n");
            }
            else if (i == data.stops.size() - 1)
            {
                Console::write(to_string(entry.arrival) + "\n");
            }
            else
            {
                Console::write(to_string(entry.arrival) + nbsp + to_string(entry.departure) + "\n");
            }
        }
        print_trip_service_days(data);
        Console::write("\n");
    }
    Console::write(std::to_string(result.size()) + " fitting entries found.\n");
}

void Trip_cmd::help()
{
    Console::write("Usage: trip <name>\n");
    Console::write("Shows all trips that fit the given name.\n");
    Console::write(
        "Since the output may be rather huge, especially for regional or local services, the name "
        "allows to use regular expression syntax. Note that if the expressions contains spaces, "
        "the whole string needs to be quoted like this: \"filter with spaces\".\n");
    Console::write("\"trip ICE.24.+\" for example will show all trips that are named ICE 24*, e.g. "
                   "ICE 245, ICE 2479\n");
}
} // namespace gtfsplanner
