#include "commands/route.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/map_writer.h"
#include "helpers/progress_writer.h"
#include "helpers/string_functions.h"

namespace gtfsplanner {
Route_cmd::Route_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}

void Route_cmd::sanitize()
{
    std::vector<std::string> parameters {"date",
                                         "start",
                                         "end",
                                         "from",
                                         "to",
                                         "total_time",
                                         "max_transfers",
                                         "min_transfer_time",
                                         "max_transfer_time",
                                         "map",
                                         "mapmode",
                                         "heatmode"};
    auto const& params = get_parameters();
    bool custom_total_time = false;
    if (!params.empty())
    {
        check_for_unknown_params(params, parameters, "route");
        check_for_duplicate_params(params);

        retrieve_param(m_date, params, "date");
        retrieve_param(m_start, params, "start");
        retrieve_param(m_end, params, "end");
        retrieve_param(m_min_transfer_time, params, "min_transfer_time");
        retrieve_param(m_max_transfer_time, params, "max_transfer_time");
        retrieve_param(m_total_time, params, "total_time");
        retrieve_param(m_max_transfers, params, "max_transfers");
        retrieve_param(m_from, params, "from");
        retrieve_param(m_to, params, "to");
        retrieve_param(m_map, params, "map");
        retrieve_param(m_mapmode, params, "mapmode");
        retrieve_param(m_heatmode, params, "heatmode");

        if (has_param(params, "end") && !has_param(params, "start"))
        {
            m_backwards_search = true;
        }

        if (has_param(params, "total_time"))
        {
            custom_total_time = true;
        }

        if (m_min_transfer_time > m_max_transfer_time)
        {
            error("Minimum transfer time must be smaller than maximum transfer time for route "
                  "command.");
        }
        if (!has_param(params, "from"))
        {
            m_backwards_search = true;
        }
        if (m_from.empty() && m_to.empty())
        {
            error(R"(At least one of "from" and "to" has to be given for route command.)");
        }
        if (m_start > m_end)
        {
            error("Start time must be lower than end time");
        }
        if (m_end - m_start < m_total_time)
        {
            if (custom_total_time)
            {
                error("Total time for route exceeds time defined by start and end parameters.");
            }
            m_total_time = m_end - m_start;
        }
        if (m_end == Time {24, 0, 0})
        {
            m_end = m_start + m_total_time;
        }
        if (m_start == Time {0, 0, 0})
        {
            m_start = m_end - m_total_time;
        }
        if (has_param(params, "map") && !m_from.empty() && !m_to.empty())
        {
            error("Map option does not work for routing with given start and end.");
        }
        check_mapmode(m_mapmode);
        check_heatmode(m_heatmode);
    }
}

struct Route_step
{
    std::string trip_id;
    std::string from_id;
    std::string to_id;

    Time start;
    Time end;
};

struct Travel_route
{
    std::vector<Route_step> steps;
    Time current_time {0, 0, 0};
    std::string current_station_id;
    gtfs::Calendar service_calendar;
    std::vector<gtfs::Calendar_date> service_exceptions;
};

std::vector<Date> get_active_days(gtfs::Calendar const& calendar,
                                  std::vector<gtfs::Calendar_date> const& calendar_dates)
{
    std::vector<Date> dates;
    for (auto date = calendar.start_date; date <= calendar.end_date; date++)
    {
        if (is_active_on_date(calendar, calendar_dates, date))
        {
            dates.push_back(date);
        }
    }
    return dates;
}

bool has_less_restrictive_calendar(Travel_route const& route,
                                   gtfs::Calendar const& calendar,
                                   std::vector<gtfs::Calendar_date> const& calendar_dates)
{
    std::vector<Date> dates = get_active_days(route.service_calendar, route.service_exceptions);
    std::vector<Date> new_dates = get_active_days(calendar, calendar_dates);
    return new_dates.size() > dates.size();
}

Time get_route_time(Travel_route const& route)
{
    if (route.steps.empty())
    {
        return Time {0, 0, 0};
    }
    auto end = route.steps.back().end;
    return end - route.steps.front().start;
}

struct Best_routes
{
    Travel_route earliest;
    Travel_route fastest;
    Travel_route least_transfers;
};

// based on implementation of std::set_intersection
template <typename InputIt1, typename InputIt2>
bool is_disjoint(InputIt1 first1, InputIt1 last1, InputIt2 first2, InputIt2 last2)
{
    while (first1 != last1 && first2 != last2)
    {
        if (*first1 < *first2)
        {
            ++first1;
        }
        else
        {
            if (!(*first2 < *first1))
            {
                return false; // *first1 and *first2 are equivalent => not disjoint.
            }
            ++first2;
        }
    }
    return true;
}

/// compare route calendar and trip calendar, if trip is not compatible (different dates), ignore this trip
/// \return true if calendars have common days, false otherwise
bool calendars_compatible(Travel_route& route, Trip_data const& trip)
{
    if (route.steps.empty())
    {
        // no route steps yet, so no constraints
        return true;
    }
    if (route.service_calendar.start_date == Date {0, 0, 0}
        || trip.service_calendar.start_date == Date {0, 0, 0})
    {
        // route or trip consists of walking to another station, that is alwas valid
        return true;
    }
    auto dates = get_active_days(route.service_calendar, route.service_exceptions);
    auto new_dates = get_active_days(trip.service_calendar, trip.service_exceptions);

    // dates and new_dates have to be sorted (they are, as get_active_days returns them in sorted order)
    return !is_disjoint(dates.begin(), dates.end(), new_dates.begin(), new_dates.end());
}

void update_calendars(gtfs::Calendar& calendar,
                      std::vector<gtfs::Calendar_date>& dates,
                      Trip_data const& trip)
{
    if (calendar.start_date == Date {0, 0, 0})
    {
        calendar = trip.service_calendar;
        dates = trip.service_exceptions;
        return;
    }
    calendar.start_date = std::max(calendar.start_date, trip.service_calendar.start_date);
    calendar.end_date = std::min(calendar.end_date, trip.service_calendar.end_date);
    calendar.monday = calendar.monday && trip.service_calendar.monday;
    calendar.tuesday = calendar.tuesday && trip.service_calendar.tuesday;
    calendar.wednesday = calendar.wednesday && trip.service_calendar.wednesday;
    calendar.thursday = calendar.thursday && trip.service_calendar.thursday;
    calendar.friday = calendar.friday && trip.service_calendar.friday;
    calendar.saturday = calendar.saturday && trip.service_calendar.saturday;
    calendar.sunday = calendar.sunday && trip.service_calendar.sunday;

    std::vector<gtfs::Calendar_date> exceptions;
    std::set_intersection(dates.begin(), dates.end(), trip.service_exceptions.begin(),
                          trip.service_exceptions.end(), std::back_inserter(exceptions),
                          [](gtfs::Calendar_date const& lhs, gtfs::Calendar_date const& rhs) {
                              return lhs.date < rhs.date && lhs.exception_type < rhs.exception_type
                                     && lhs.service_id < rhs.service_id;
                          });
    dates = exceptions;
}

void update_calendars(Travel_route& route, Trip_data const& trip)
{
    if (route.steps.empty())
    {
        route.service_calendar = trip.service_calendar;
        route.service_exceptions = trip.service_exceptions;
    }
    else
    {
        update_calendars(route.service_calendar, route.service_exceptions, trip);
    }
}

void print_route(gtfs::Dataset const& dataset, std::vector<Travel_route> const& routes)
{
    for (auto i = 0U; i < routes.size(); i++)
    {
        if (i > 0)
        {
            Console::write("--------------------\n");
        }
        auto end = routes[i].steps.back().end;
        auto start = routes[i].steps[0].start;
        auto time = end - start;
        auto const& route_steps = routes[i].steps;
        Console::write("Option " + std::to_string(i + 1) + ": "
                       + std::to_string(route_steps.size() - 1) + " transfers, " + to_string(time)
                       + " total time\n");
        for (auto j = 0U; j < route_steps.size(); j++)
        {
            auto const& step = route_steps[j];
            auto from_idx = dataset.stop_id_to_idx.at(step.from_id);
            auto to_idx = dataset.stop_id_to_idx.at(step.to_id);
            auto const& from = dataset.stops[from_idx];
            auto const& to = dataset.stops[to_idx];
            std::string trip_name = "Walking";
            if (!step.trip_id.empty())
            {
                auto trip_idx = dataset.trip_id_to_idx.at(step.trip_id);
                trip_name = build_route_name(dataset, dataset.trips[trip_idx]);
            }
            Console::write("Step " + std::to_string(j + 1) + "\t" + trip_name + "\n");
            Console::write(to_string(step.start) + " " + from.name + "\n");
            Console::write(to_string(step.end) + " " + to.name + "\n");
            Console::write("\n");
        }
    }
}

bool trip_transfer_visited(
    std::unordered_map<std::pair<std::string, std::string>, std::string, pair_hash> const&
        trip_transfers_visited,
    std::vector<Route_step> const& steps,
    std::string const& trip_id,
    bool backwards_search)
{
    if (steps.empty())
    {
        return false;
    }
    auto const& current_trip_id = (backwards_search) ? steps.front().trip_id : steps.back().trip_id;
    auto iter = trip_transfers_visited.find(std::make_pair(current_trip_id, trip_id));
    if (iter == trip_transfers_visited.end())
    {
        return false;
    }
    // we have a transfer for these two trips, check if it would be the same station
    // if it is the same station then there's no problem
    if (backwards_search)
    {
        auto const& station = steps.front().from_id;
        return station != iter->second;
    }
    auto const& station = steps.back().to_id;
    return station != iter->second;
}

bool init_stop_id(std::string& id, std::string const& name, gtfs::Dataset const& dataset)
{
    if (!name.empty())
    {
        auto iter = find_station(dataset, name);
        if (iter != dataset.stops.end())
        {
            id = iter->id;
            return true;
        }
        return false;
    }
    return true;
}

void init_queue(std::deque<Travel_route>& route_queue,
                std::string const& from_id,
                std::string const& to_id,
                Time const& t_start,
                Time const& t_end,
                bool backwards_search)
{
    Travel_route start;
    if (!backwards_search)
    {
        start.current_time = t_start;
        start.current_station_id = from_id;
    }
    else
    {
        start.current_time = t_end;
        start.current_station_id = to_id;
    }
    route_queue.push_back(start);
}

Time calculate_next_time(Time const& current_time,
                         bool add_transfer_time,
                         Time const& transfer_time,
                         bool backwards_search)
{
    auto next_time = current_time;
    if (add_transfer_time)
    {
        if (!backwards_search)
        {
            // add time for transfer before searching for next departures
            next_time += transfer_time;
        }
        else
        {
            // remove time for transfer before searching for previous arrivals
            if (transfer_time > next_time)
            {
                next_time.hour += 24;
            }
            next_time -= transfer_time;
        }
    }
    return next_time;
}

template <typename T, typename U>
void init_best_iterators(T& routes,
                         U& fastest,
                         U& least_transfers,
                         U& most_exact,
                         Time const& target_time,
                         bool backwards_search)
{
    fastest = std::min_element(
        routes.begin(), routes.end(),
        [backwards_search, target_time](Travel_route const& lhs, Travel_route const& rhs) {
            auto l_time = get_route_time(lhs);
            auto r_time = get_route_time(rhs);
            if (l_time != r_time)
            {
                return l_time < r_time;
            }
            if (backwards_search)
            {
                return target_time - lhs.steps.back().end > target_time - rhs.steps.back().end;
            }
            return lhs.current_time - target_time < rhs.current_time - target_time;
        });
    least_transfers = std::min_element(routes.begin(), routes.end(),
                                       [](Travel_route const& lhs, Travel_route const& rhs) {
                                           if (lhs.steps.size() != rhs.steps.size())
                                           {
                                               return lhs.steps.size() < rhs.steps.size();
                                           }
                                           return get_route_time(lhs) < get_route_time(rhs);
                                       });
    most_exact = std::min_element(
        routes.begin(), routes.end(),
        [backwards_search, target_time](Travel_route const& lhs, Travel_route const& rhs) {
            if (backwards_search)
            {
                return target_time - lhs.steps.back().end < target_time - rhs.steps.back().end;
            }
            return lhs.steps.front().start - target_time < rhs.steps.front().start - target_time;
        });
}

template <typename T, typename U>
void init_best_values(T& routes,
                      U& fastest,
                      U& least_transfers,
                      U& most_exact,
                      Time& best_time,
                      size_t& best_transfers,
                      Time& best_exactness,
                      Time& max_allowed_time_for_best_transfer,
                      Time& max_allowed_time_for_most_exact,
                      Time const& target_time,
                      bool backwards_search)
{
    init_best_iterators(routes, fastest, least_transfers, most_exact, target_time,
                        backwards_search);
    best_time = get_route_time(*fastest);
    best_transfers = least_transfers->steps.size();
    auto time = (backwards_search) ? most_exact->steps.back().end : target_time;
    if (time > target_time && backwards_search)
    {
        time.hour -= 24;
    }
    best_exactness = (backwards_search) ? target_time - time : time - target_time;
    max_allowed_time_for_best_transfer =
        std::min(Time {3, 0, 0} + best_time, best_time + best_time);
    max_allowed_time_for_most_exact = std::min(Time {2, 0, 0} + best_time, best_time + best_time);
}

bool is_useful_route(Travel_route const& current_route,
                     Route_step const& next_step,
                     gtfs::Calendar const& calendar,
                     std::vector<gtfs::Calendar_date> const& calendar_dates,
                     std::unordered_map<std::string, std::vector<Travel_route>> const& best_routes,
                     Time const& target_time,
                     bool backwards_search)
{
    auto iter = (backwards_search) ? best_routes.find(next_step.from_id)
                                   : best_routes.find(next_step.to_id);
    if (iter == best_routes.end())
    {
        // no route yet to this station, it is useful to follow this step
        return true;
    }
    // one or more routes to the current destination exist, check according to following criteria
    // a) fastest route - the one with the shortest time
    // b) fewest transfers - the one with fewest transfers, but may be slower
    // c) earliest/latest route - the one that allows the earliest start (!backwards_search) or the latest arrival(backwards_search)

    // Example: routing from A to B start 08:00:00 end 16:00:00
    // a) departs at 9:30 and arrives at 11:30 with one transfer (2h route time)
    // b) departs at 10:30 and arrives at 13:00 with no transfers (2:30h route time)
    // c) departs at 8:30 and arrives at 11:00 with two transfers (2:30h route time, but starts closest to 08:00)

    // Example: routing from A to B end 16:00:00 => backwards search
    // a) departs at 9:30 and arrives at 11:30 with one transfer (2h route time)
    // b) departs at 10:30 and arrives at 13:00 with no transfers (2:30h route time)
    // c) departs at 12:30 and arrives at 15:00 with two transfers (2:30h route time, but arrives closest to 16:00)

    // arbitrary constants to rule out various routes that are completely idiotic (large detours and such):
    // a route that has fewer transfers may take 3h longer, or twice the fastest time, whatever is less
    // a route may have up to three additional transfers compared to the route with the fewest transfers
    // the route closest to target time may take up to 2h longer, or twice the fastest time, whatever is less
    // the route closest to target time may have up to four additional transfers compared to the one with the fewest transfers
    auto const& routes = iter->second;

    std::vector<Travel_route>::const_iterator fastest;
    std::vector<Travel_route>::const_iterator least_transfers;
    std::vector<Travel_route>::const_iterator most_exact;
    Time best_time {0, 0, 0};
    size_t best_transfers;
    Time best_exactness {0, 0, 0};
    Time max_allowed_time_for_best_transfer {0, 0, 0};
    Time max_allowed_time_for_most_exact {0, 0, 0};
    init_best_values(routes, fastest, least_transfers, most_exact, best_time, best_transfers,
                     best_exactness, max_allowed_time_for_best_transfer,
                     max_allowed_time_for_most_exact, target_time, backwards_search);

    auto num_transfers = current_route.steps.size() + 1; // +1 for the transfer we're going to add
    auto step_end = next_step.end;
    auto step_start = next_step.start;
    auto time = step_end - step_start;
    if (!current_route.steps.empty())
    {
        if (backwards_search)
        {
            step_end = current_route.steps.back().end;
            time = step_end - next_step.start;
        }
        else
        {
            step_end = next_step.end;
            time = next_step.end - current_route.steps.front().start;
        }
    }
    auto deviation =
        (backwards_search) ? target_time - next_step.end : next_step.start - target_time;
    if (!current_route.steps.empty())
    {
        deviation = (backwards_search) ? target_time - current_route.steps.back().end
                                       : current_route.steps.front().start - target_time;
    }

    // now compare the values of the best and the current routes
    if (best_time > time && num_transfers <= best_transfers + 3 && deviation <= Time {4, 0, 0})
    {
        // route is faster
        return true;
    }
    if (best_transfers > num_transfers && time <= max_allowed_time_for_best_transfer
        && deviation <= Time {4, 0, 0})
    {
        // route has fewer transfers
        return true;
    }
    if (best_exactness > deviation && time <= max_allowed_time_for_most_exact
        && num_transfers <= best_transfers + 4)
    {
        return true;
    }

    // special case: route has identical properties => typically only different service days, treat it as useful and
    // compare calendars later
    if (best_time == time && num_transfers == best_transfers && best_exactness == deviation)
    {
        return has_less_restrictive_calendar(*fastest, calendar, calendar_dates)
               || has_less_restrictive_calendar(*least_transfers, calendar, calendar_dates)
               || has_less_restrictive_calendar(*most_exact, calendar, calendar_dates);
    }
    return false;
}

void update_route(Travel_route& route, Travel_route const& new_route)
{
    route.current_time = new_route.current_time;
    route.service_calendar = new_route.service_calendar;
    route.service_exceptions = new_route.service_exceptions;
    route.steps = new_route.steps;
}

void update_best_routes(std::unordered_map<std::string, std::vector<Travel_route>>& best_routes,
                        Travel_route const& route,
                        Time const& target_time,
                        bool backwards_search)
{
    auto iter = best_routes.find(route.current_station_id);
    if (iter == best_routes.end())
    {
        // no route yet, just insert the new one, it automatically is the best
        best_routes.insert(
            std::make_pair(route.current_station_id, std::vector<Travel_route> {route}));
        return;
    }
    // one or more routes to the current destination exist, add/replace them according to rating
    // one or more routes to the current destination exist, check according to following criteria
    // a) fastest route - the one with the shortest time
    // b) fewest transfers - the one with fewest transfers, but may be slower
    // c) earliest/latest route - the one that allows the earliest start (!backwards_search) or the latest arrival(backwards_search)

    // Example: routing from A to B start 08:00:00 end 16:00:00
    // a) departs at 9:30 and arrives at 11:30 with one transfer (2h route time)
    // b) departs at 10:30 and arrives at 13:00 with no transfers (2:30h route time)
    // c) departs at 8:30 and arrives at 11:00 with two transfers (2:30h route time, but starts closest to 08:00)

    // Example: routing from A to B end 16:00:00 => backwards search
    // a) departs at 9:30 and arrives at 11:30 with one transfer (2h route time)
    // b) departs at 10:30 and arrives at 13:00 with no transfers (2:30h route time)
    // c) departs at 12:30 and arrives at 15:00 with two transfers (2:30h route time, but arrives closest to 16:00)

    // arbitrary constants to rule out various routes that are completely idiotic (large detours and such):
    // a route that has fewer transfers may take 3h longer, or twice the fastest time, whatever is less
    // a route may have up to three additional transfers compared to the route with the fewest transfers
    // the route closest to target time may take up to 2h longer, or twice the fastest time, whatever is less
    // the route closest to target time may have up to four additional transfers compared to the one with the fewest transfers
    auto& routes = iter->second;
    std::vector<Travel_route>::iterator fastest;
    std::vector<Travel_route>::iterator least_transfers;
    std::vector<Travel_route>::iterator most_exact;
    Time best_time {0, 0, 0};
    size_t best_transfers;
    Time best_exactness {0, 0, 0};
    Time max_allowed_time_for_best_transfer {0, 0, 0};
    Time max_allowed_time_for_most_exact {0, 0, 0};
    init_best_values(routes, fastest, least_transfers, most_exact, best_time, best_transfers,
                     best_exactness, max_allowed_time_for_best_transfer,
                     max_allowed_time_for_most_exact, target_time, backwards_search);

    auto num_transfers = route.steps.size(); // +1 for the transfer we're going to add
    auto time = get_route_time(route);
    auto last_step_time = (backwards_search) ? route.steps.back().end : route.steps.front().start;
    if (last_step_time > target_time && backwards_search)
    {
        last_step_time.hour -= 24;
    }
    auto deviation =
        (backwards_search) ? target_time - last_step_time : last_step_time - target_time;
    bool updated = false;
    // now compare the values of the best and the current routes
    if (best_time > time && num_transfers <= best_transfers + 3 && deviation <= Time {4, 0, 0})
    {
        // new route is faster
        if ((num_transfers > best_transfers && fastest == least_transfers)
            || (deviation > best_exactness && fastest == most_exact))
        {
            routes.push_back(route);
            init_best_values(routes, fastest, least_transfers, most_exact, best_time,
                             best_transfers, best_exactness, max_allowed_time_for_best_transfer,
                             max_allowed_time_for_most_exact, target_time, backwards_search);
        }
        else
        {
            update_route(*fastest, route);
        }
        updated = true;
    }
    if (best_transfers > num_transfers && time <= max_allowed_time_for_best_transfer
        && deviation <= Time {4, 0, 0})
    {
        // route has fewer transfers
        if ((time > best_time && fastest == least_transfers)
            || (deviation > best_exactness && least_transfers == most_exact))
        {
            routes.push_back(route);
            init_best_values(routes, fastest, least_transfers, most_exact, best_time,
                             best_transfers, best_exactness, max_allowed_time_for_best_transfer,
                             max_allowed_time_for_most_exact, target_time, backwards_search);
        }
        else
        {
            update_route(*least_transfers, route);
        }
        updated = true;
    }
    if (best_exactness > deviation && time <= max_allowed_time_for_most_exact
        && num_transfers <= best_transfers + 4)
    {
        if ((time > best_time && most_exact == fastest)
            || (num_transfers > best_transfers && most_exact == least_transfers))
        {
            routes.push_back(route);
            init_best_values(routes, fastest, least_transfers, most_exact, best_time,
                             best_transfers, best_exactness, max_allowed_time_for_best_transfer,
                             max_allowed_time_for_most_exact, target_time, backwards_search);
        }
        else
        {
            update_route(*most_exact, route);
        }
        updated = true;
    }

    // special case: route has identical properties => typically only different service days, treat it as useful and
    // compare calendars later
    if (!updated && best_time == time && num_transfers == best_transfers
        && best_exactness == deviation)
    {
        if (has_less_restrictive_calendar(*fastest, route.service_calendar,
                                          route.service_exceptions))
        {
            update_route(*fastest, route);
        }
        if (has_less_restrictive_calendar(*least_transfers, route.service_calendar,
                                          route.service_exceptions))
        {
            update_route(*least_transfers, route);
        }
        if (has_less_restrictive_calendar(*most_exact, route.service_calendar,
                                          route.service_exceptions))
        {
            update_route(*most_exact, route);
        }
    }
}

bool route_time_exceeded(Travel_route const& current_route,
                         Route_step const& next_step,
                         Time const& start,
                         Time const& end,
                         Time const& total_time)
{
    auto step_end = next_step.end;
    auto step_start = next_step.start;
    auto elapsed_time = step_end - step_start;
    if (!current_route.steps.empty())
    {
        auto const& first_step = current_route.steps.front();
        auto const& last_step = current_route.steps.back();
        elapsed_time = elapsed_time + (last_step.end - first_step.start);
    }
    return (elapsed_time > total_time || step_end > end || step_start < start);
}

void Route_cmd::add_transfers_to_queue(
    std::deque<Travel_route>& route_queue,
    Travel_route const& current_route,
    std::unordered_set<std::string> const& visited,
    std::unordered_map<std::string, std::vector<Travel_route>>& best_routes,
    gtfs::Dataset const& dataset)
{
    auto start = std::lower_bound(
        dataset.transfers.begin(), dataset.transfers.end(), current_route.current_station_id,
        [](gtfs::Transfer const& transfer, std::string const& current_id) {
            return transfer.from_stop_id < current_id;
        });
    auto end = std::upper_bound(dataset.transfers.begin(), dataset.transfers.end(),
                                current_route.current_station_id,
                                [](std::string const& current_id, gtfs::Transfer const& transfer) {
                                    return current_id < transfer.from_stop_id;
                                });
    for (auto iter = start; iter != end; ++iter)
    {
        auto const& transfer = *iter;
        if (visited.find(transfer.to_stop_id) != visited.end())
        {
            continue;
        }
        auto transfer_time =
            std::max(Time {0, 0, transfer.min_transfer_time_sec}, m_min_transfer_time);
        auto next_time = calculate_next_time(current_route.current_time, true, transfer_time,
                                             m_backwards_search);
        auto next_step = Route_step {"", transfer.from_stop_id, transfer.to_stop_id,
                                     current_route.current_time, next_time};
        if (m_backwards_search)
        {
            std::swap(next_step.start, next_step.end);
            std::swap(next_step.from_id, next_step.from_id);
        }

        if (route_time_exceeded(current_route, next_step, m_start, m_end, m_total_time))
        {
            continue;
        }
        gtfs::Calendar calendar;
        std::vector<gtfs::Calendar_date> calendar_dates;
        // only add queue if it will improve results
        if (is_useful_route(current_route, next_step, calendar, calendar_dates, best_routes,
                            (m_backwards_search) ? m_end : m_start, m_backwards_search))
        {
            route_queue.push_back(current_route);
            auto& updated_route = route_queue.back();
            if (m_backwards_search)
            {
                updated_route.steps.insert(updated_route.steps.begin(), next_step);
            }
            else
            {
                updated_route.steps.push_back(next_step);
            }
            updated_route.current_station_id = transfer.to_stop_id;
            updated_route.current_time = next_time;
            update_best_routes(best_routes, updated_route, (m_backwards_search) ? m_end : m_start,
                               m_backwards_search);
        }
    }
}

std::vector<Trip_data> get_next_steps(gtfs::Dataset const& dataset,
                                      std::string const& station_id,
                                      Date const& date,
                                      Time const& start,
                                      Time const& end,
                                      bool backwards_search)
{
    std::vector<Trip_data> next_steps;
    if (backwards_search)
    {
        next_steps = get_arrivals(dataset, station_id, date, end, start);
        // arrivals are ordered by arrival time, but we want to search backwards, starting with the last arrivals
        std::reverse(next_steps.begin(), next_steps.end());
    }
    else
    {
        next_steps = get_departures(dataset, station_id, date, start, end);
    }
    return next_steps;
}

void update_trip_transfer_visited(
    std::unordered_map<std::pair<std::string, std::string>, std::string, pair_hash>&
        trip_transfers_visited,
    Travel_route const& route,
    bool backwards_search)
{
    if (route.steps.size() > 1)
    {
        // the relevant transfer depends on the direction
        if (backwards_search)
        {
            auto from_trip = route.steps[1];
            auto to_trip = route.steps[0];
            auto key = std::make_pair(from_trip.trip_id, to_trip.trip_id);
            trip_transfers_visited.insert(std::make_pair(key, to_trip.to_id));
        }
        else
        {
            auto from_trip = route.steps[route.steps.size() - 2];
            auto to_trip = route.steps[route.steps.size() - 1];
            auto key = std::make_pair(from_trip.trip_id, to_trip.trip_id);
            trip_transfers_visited.insert(std::make_pair(key, to_trip.from_id));
        }
    }
}

void Route_cmd::execute(gtfs::Dataset& dataset)
{
    std::unordered_set<std::string> visited; // the cities visited
    std::deque<Travel_route> route_queue;
    // from,to - switch at
    std::unordered_map<std::pair<std::string, std::string>, std::string, pair_hash>
        trip_transfers_visited;
    std::unordered_map<std::string, std::vector<Travel_route>> best_routes;
    std::string to_id;
    std::string from_id;

    if (!init_stop_id(to_id, m_to, dataset) || !init_stop_id(from_id, m_from, dataset))
    {
        // station not found, cannot do route calculation
        return;
    }

    init_queue(route_queue, from_id, to_id, m_start, m_end, m_backwards_search);

    // calculation progress: track the time in the route queue in terms of total time to track:
    // e.g. routing between 8:00 and 10:00 creates 120 minutes equaling 100%
    // the routing queue is ordered by time and therefore can be used as a percentage
    auto total = m_end - m_start;
    auto total_sec = total.hour * 3600 + total.minute * 60 + total.second;
    Progress_writer progress("Calculating route", total_sec);
    auto current_time = route_queue.front().current_time;
    while (!route_queue.empty())
    {
        auto current_route = route_queue.front();
        route_queue.pop_front();
        auto vis_iter = visited.insert(current_route.current_station_id);
        // uncomment this for debugging, it shows the visited stops and the best times for each of them
        /*if (vis_iter.second)
        {
            auto idx = dataset.stop_id_to_idx.find(current_route.current_station_id);
            auto const& stop = dataset.stops[idx->second];
            std::stringstream ss;
            ss << current_route.current_time;
            Console::write(stop.name + " " + ss.str() + "\n");
        }*/

        if (current_time != current_route.current_time)
        {
            // current_route.current_time may be larger than the end time, this will result in
            // "route time exceeded", but that's not yet checked here
            if (m_backwards_search && current_time > current_route.current_time)
            {
                auto diff = (m_backwards_search) ? current_time - current_route.current_time
                                                 : current_route.current_time - current_time;
                auto diff_s = diff.hour * 3600 + diff.minute * 60 + diff.second;
                progress.update(diff_s);
                current_time = current_route.current_time;
            }
        }

        if (current_route.steps.size() == m_max_transfers + 1)
        {
            // maximum defined number of transfers reach, stop here
            continue;
        }
        auto next_time =
            calculate_next_time(current_route.current_time, !current_route.steps.empty(),
                                m_min_transfer_time, m_backwards_search);

        // follow transfers on foot if possible
        add_transfers_to_queue(route_queue, current_route, visited, best_routes, dataset);
        auto latest_next_time = (m_backwards_search) ? m_start : m_end;
        auto next_potential_steps =
            get_next_steps(dataset, current_route.current_station_id, m_date, next_time,
                           latest_next_time, m_backwards_search);

        // next_potential_steps contains everything that arrives/departs from current station ordered by time,
        // iterate over each of the trips and add a routestep for each of the remaining destinations,
        // unless it would violate constraints or would not improve the results
        for (auto const& next : next_potential_steps)
        {
            // if calendars are not compatible, ignore - incompatible calendars means e.g. trip A
            // running mo-fr and trip B running sa-su or only on disjunct dates
            if (!calendars_compatible(current_route, next))
            {
                continue;
            }

            // only check waiting time if we actually did some kind of trip, if there was no step until now we can just
            // start the journey at another point in time without waiting, same if the first step of the journey was
            // "walking to another station"
            if (!current_route.steps.empty()
                && !(current_route.steps.size() == 1
                     && current_route.steps.front().trip_id.empty()))
            {
                auto start_time =
                    (m_backwards_search) ? next.stops.back().arrival : next.stops.front().departure;
                while (start_time.hour >= 24)
                {
                    start_time.hour -= 24;
                }
                if (m_backwards_search)
                {
                    auto end = current_route.current_time;
                    auto start = start_time;
                    if (end < start)
                    {
                        end.hour += 24;
                    }
                    auto waiting_time = end - start;
                    if (waiting_time > m_max_transfer_time)
                    {
                        continue;
                    }
                }
                else
                {
                    // what to do at midnight?
                    if (start_time < current_route.current_time)
                    {
                        start_time.hour += 24;
                    }
                    auto waiting_time = start_time - current_route.current_time;
                    if (waiting_time > m_max_transfer_time)
                    {
                        continue;
                    }
                }
            }

            // calculate the updated calendar containing only the days that are valid when the current trip
            // is added to the route (need this info to compare if the route quality is better)
            gtfs::Calendar calendar(current_route.service_calendar);
            std::vector<gtfs::Calendar_date> calendar_dates(current_route.service_exceptions);
            update_calendars(calendar, calendar_dates, next);

            // now iterate over all stations of the potential trip and add an updated route to the
            // route_queue
            for (auto const& station : next.stops)
            {
                // if the transfer from the last trip to this trip has already been done, skip.
                // this prevents situations like these:
                // trip 1 with stops A B C D and trip 2 with stops E B C F
                // B and C are both valid places for transfer, but it would produce duplicate results
                // only transfer at station B, then skip any further transfers between these two trips
                if (trip_transfer_visited(trip_transfers_visited, current_route.steps, next.trip_id,
                                          m_backwards_search))
                {
                    continue;
                }

                // if the station has already been visited, it can be skipped - by using a priority queue
                // the best results have already been worked on
                if (visited.find(station.stop_id) != visited.end())
                {
                    continue;
                }

                auto start_time =
                    (m_backwards_search) ? next.stops.back().arrival : next.stops.front().departure;
                auto end_time = (m_backwards_search) ? station.departure : station.arrival;
                Route_step next_step {next.trip_id, current_route.current_station_id,
                                      station.stop_id, start_time, end_time};
                if (next_step.start.hour >= 24 && next_step.end.hour >= 24)
                {
                    next_step.start.hour -= 24;
                    next_step.end.hour -= 24;
                }
                if (m_backwards_search)
                {
                    std::swap(next_step.from_id, next_step.to_id);
                    std::swap(next_step.start, next_step.end);
                }
                if (next_step.start > next_step.end)
                {
                    // trip over midnight
                    next_step.end.hour += 24;
                }
                if (route_time_exceeded(current_route, next_step, m_start, m_end, m_total_time))
                {
                    continue;
                }

                // only add to the queue if this will actually improve results
                if (is_useful_route(current_route, next_step, calendar, calendar_dates, best_routes,
                                    (m_backwards_search) ? m_end : m_start, m_backwards_search))
                {
                    route_queue.push_back(current_route);
                    auto& updated_route = route_queue.back();
                    update_calendars(updated_route, next);
                    // update walking time if there is any as the only previous step
                    if (updated_route.steps.size() == 1
                        && updated_route.steps.front().trip_id.empty())
                    {
                        if (m_backwards_search)
                        {
                            auto& walk = updated_route.steps.front();
                            auto walktime = walk.end - walk.start;
                            walk.start = next_step.end;
                            walk.end = next_step.end + walktime;
                        }
                        else
                        {
                            auto& walk = updated_route.steps.front();
                            auto walktime = walk.end - walk.start;
                            walk.start = next_step.start - walktime;
                            walk.end = next_step.start;
                        }
                    }
                    if (m_backwards_search)
                    {
                        updated_route.steps.insert(updated_route.steps.begin(), next_step);
                    }
                    else
                    {
                        updated_route.steps.push_back(next_step);
                    }
                    auto time = updated_route.steps.front().start;
                    for (auto& step : updated_route.steps)
                    {
                        if (step.start < time)
                        {
                            step.start.hour += 24;
                            step.end.hour += 24;
                        }
                        time = step.end;
                    }
                    updated_route.current_station_id = station.stop_id;
                    updated_route.current_time = end_time;

                    update_trip_transfer_visited(trip_transfers_visited, updated_route,
                                                 m_backwards_search);
                    update_best_routes(best_routes, updated_route,
                                       (m_backwards_search) ? m_end : m_start, m_backwards_search);
                }
            }
        }

        // sort the route queue according to time stamp and number of steps
        std::sort(route_queue.begin(), route_queue.end(),
                  [this](Travel_route const& lhs, Travel_route const& rhs) {
                      if (lhs.current_time != rhs.current_time)
                      {
                          auto time_l = lhs.current_time;
                          while (time_l.hour >= 24)
                          {
                              time_l.hour -= 24;
                          }
                          auto time_r = rhs.current_time;
                          while (time_r.hour >= 24)
                          {
                              time_r.hour -= 24;
                          }
                          if (m_backwards_search)
                          {
                              return time_l > time_r;
                          }
                          return time_l < time_r;
                      }
                      return lhs.steps.size() < rhs.steps.size();
                  });
    }

    if (!m_to.empty() && !m_from.empty())
    {
        auto best = best_routes.end();
        if (m_backwards_search)
        {
            best = best_routes.find(from_id);
        }
        else
        {
            best = best_routes.find(to_id);
        }

        if (best != best_routes.end())
        {
            print_route(dataset, best->second);
        }
        else
        {
            Console::write(
                "Could not find any route to destination which fits the given parameters.\n");
        }
    }
    if (m_to.empty() || m_from.empty())
    {
        // show routes to all found places
        // retrieve best trips and sort by destination
        std::vector<std::pair<std::string, std::vector<Travel_route>>> all_trips;
        all_trips.reserve(best_routes.size());
        for (auto const& destination : best_routes)
        {
            auto stop_iter = dataset.stop_id_to_idx.find(destination.first);
            check_iter(stop_iter, dataset.stop_id_to_idx);
            auto const& stop = dataset.stops[stop_iter->second];
            all_trips.emplace_back(std::make_pair(stop.name, destination.second));
        }
        // sort by name
        std::sort(all_trips.begin(), all_trips.end(),
                  [](std::pair<std::string, std::vector<Travel_route>> const& lhs,
                     std::pair<std::string, std::vector<Travel_route>> const& rhs) {
                      return lhs.first < rhs.first;
                  });
        for (auto const& destination : all_trips)
        {
            Console::write(to_upper(destination.first) + "\n");
            print_route(dataset, destination.second);
        }

        if (!m_map.empty())
        {
            std::unordered_map<std::string, Time> count;
            for (auto const& route : best_routes)
            {
                // find the route that is the one with the shortest time
                auto iter =
                    std::min_element(route.second.begin(), route.second.end(),
                                     [this](Travel_route const& lhs, Travel_route const& rhs) {
                                         if (m_backwards_search)
                                         {
                                             return lhs.current_time > rhs.current_time;
                                         }
                                         return lhs.current_time < rhs.current_time;
                                     });
                auto time = get_route_time(*iter);
                count.insert(std::make_pair(route.first, time));
            }
            write_heatmap(dataset, m_map, m_mapmode, m_heatmode, count);
        }
    }
}

void Route_cmd::help()
{
    Console::write("Usage: route [from <stop>] [to <stop>] [date <date>] [start <time>] [end "
                   "<time>] [total_time <time>] [max_transfers <num>] [min_transfer_time <time>] "
                   "[map <file>] [mapmode <mode>] [heatmode <mode>]\n");
    Console::write("Calculate route(s) between the given from and to stops, including transfers. "
                   "Note that either \"from\" or \"to\" has to be given.\n");
    Console::write("from: Start stop of the route.\n");
    Console::write("to: End stop of the route.\n");
    Console::write("date (default: none): only show routes for the given date.\n");
    Console::write("start (default: 00:00:00): only show routes departing at from stop equal to or "
                   "later than the given time.\n");
    Console::write("end (default: 24:00:00): only show trips arriving at to stop than or equal to "
                   "the given time.\n");
    Console::write("route_time (default: 24:00:00 or end-start): Maximum time that the route is "
                   "allowed to take.\n");
    Console::write("max_transfers (default: 10): only calculate routes with at most this many "
                   "transfers between trips.\n");
    Console::write("total_time (default: 24:00:00): only calculate routes that take at most this "
                   "much time.\n");
    Console::write("max_transfer_time (default 01:00:00): calculate routes where each transfer "
                   "between vehicles takes at most this much time.\n");
    Console::write("min_transfer_time (default 00:10:00): calculate routes where each transfer "
                   "between vehicles takes at least this much time.\n");
    Console::write(
        "map: Writes a file marking the stations that can be reached with the current route search "
        "parameters. Only valid if either from or to are given, but not both.\n");
    Console::write("mapmode (default: kml): Defines the output format. Only relevant when the map "
                   "option is used. Known formats: kml (for use in Google Earth), geojson (for use "
                   "in OpenStreetMap tools)\n");
    Console::write("heatmode (default: rainbow): Defines the color palette for the heatmap. Only "
                   "relevant when the map option is used. Known formats: inferno, monochrome "
                   "(black-white), rainbow, viridis, wistia\n");
    Console::write("Routes are ordered by departure time and rated by the number of transfers "
                   "(less is better) and travel time (less is better), as well as exactness to "
                   "given start and end times.");
    Console::write("If both start and end are given in addition to total_time, total_time has to "
                   "describe a time interval within the start-to-end range.");
    Console::write(
        "If both \"from\" and \"to\" are given, routes are calculated between both stops. If only "
        "\"from\" is given, routes to all destinations"
        "are calculated, as long as they fit the other (time) criteria. If only \"to\" is given, "
        "routes from there back to all possible "
        "departure stations are calculated, as long as they fit the other (time) criteria.\n");
    Console::write("If multiple transfers between the same trips are possible (e.g. changing "
                   "between two trains that both stop at Berlin Hbf and Berlin Spandau), "
                   "only the first possible exchange stop is considered.");
}
} // namespace gtfsplanner
