#include "commands/parser.h"

#include "commands/arrival.h"
#include "commands/count.h"
#include "commands/departure.h"
#include "commands/exit.h"
#include "commands/help.h"
#include "commands/info.h"
#include "commands/list.h"
#include "commands/load.h"
#include "commands/mapcount.h"
#include "commands/route.h"
#include "commands/trip.h"
#include "helpers/console.h"
#include "helpers/exception.h"
#include "helpers/string_functions.h"

#include <fstream>
#include <unordered_map>
#include <unordered_set>

namespace gtfsplanner {
std::vector<std::unique_ptr<Command>> load_command_file(std::string const& name)
{
    std::ifstream input(name);
    if (!input.is_open())
    {
        throw Parse_error("Could not open config file " + name);
    }
    std::vector<std::unique_ptr<Command>> result;
    std::string line;
    while (std::getline(input, line))
    {
        result.push_back(parse_command(line));
    }
    return result;
}

std::vector<std::unique_ptr<Command>> parse(std::vector<std::string> const& args)
{
    // allowed arguments:
    // --help
    // --cfg <config-file>
    std::vector<std::unique_ptr<Command>> result;

    if (!args.empty())
    {
        if (args[0] == "--help")
        {
            result.push_back(std::make_unique<Help_cmd>(
                std::vector<std::string> {args.begin() + 1, args.end()}));
        }
        else if (args[0] == "--cfg")
        {
            if (args.size() < 2)
            {
                throw Parse_error("Missing argument <file_name> for --cfg.");
            }
            return load_command_file(args[1]);
        }
        else
        {
            throw Parse_error("Unexpected argument " + args[0]);
        }
    }

    return result;
}

std::unique_ptr<Command> build_command(std::string const& cmd,
                                       std::vector<std::string> const& parameters)
{
    // the map holds empty prototypes of the command
    std::unordered_map<std::string, Commands> commands = {{"arrival", Commands::ARRIVAL},
                                                          {"count", Commands::COUNT},
                                                          {"departure", Commands::DEPARTURE},
                                                          {"exit", Commands::EXIT},
                                                          {"info", Commands::INFO},
                                                          {"help", Commands::HELP},
                                                          {"list", Commands::LIST},
                                                          {"load", Commands::LOAD},
                                                          {"mapcount", Commands::MAPCOUNT},
                                                          {"route", Commands::ROUTE},
                                                          {"trip", Commands::TRIP}};
    auto iter = commands.find(cmd);
    if (iter != commands.end())
    {
        switch (iter->second)
        {
        case Commands::ARRIVAL:
            return std::make_unique<Arrival_cmd>(parameters);
        case Commands::COUNT:
            return std::make_unique<Count_cmd>(parameters);
        case Commands::DEPARTURE:
            return std::make_unique<Departure_cmd>(parameters);
        case Commands::EXIT:
            return std::make_unique<Exit_cmd>(parameters);
        case Commands::INFO:
            return std::make_unique<Info_cmd>(parameters);
        case Commands::HELP:
            return std::make_unique<Help_cmd>(parameters);
        case Commands::LOAD:
            return std::make_unique<Load_cmd>(parameters);
        case Commands::LIST:
            return std::make_unique<List_cmd>(parameters);
        case Commands::MAPCOUNT:
            return std::make_unique<Mapcount_cmd>(parameters);
        case Commands::ROUTE:
            return std::make_unique<Route_cmd>(parameters);
        case Commands::TRIP:
            return std::make_unique<Trip_cmd>(parameters);
        }
    }
    else
    {
        Console::write("Could not parse command " + cmd + "\n");
    }
    return nullptr;
}

std::unique_ptr<Command> parse_command(std::string const& input)
{
    // tokenize the input
    std::vector<std::string> tokens = tokenize_with_quotes(sanitize_whitespaces(input), ' ');
    if (!tokens.empty())
    {
        auto parameters = std::vector<std::string>({tokens.begin() + 1, tokens.end()});
        return build_command(tokens[0], parameters);
    }
    return nullptr;
}

std::string cmds_to_string(std::vector<std::string> const& commands)
{
    std::stringstream ss;
    for (auto i = 0U; i < commands.size(); i++)
    {
        auto const& cmd = commands[i];
        if (i > 0 && i < commands.size() - 1)
        {
            ss << ", ";
        }
        if (i == commands.size() - 1)
        {
            ss << " and ";
        }
        ss << "\"" << cmd << "\"";
    }
    return ss.str();
}

void check_for_unknown_params(std::vector<std::string> const& params,
                              std::vector<std::string> const& allowed,
                              std::string const& command_name)
{
    bool valid = true;
    for (auto i = 0U; i < params.size(); i += 2)
    {
        if (std::find(allowed.begin(), allowed.end(), params[i]) == allowed.end())
        {
            Console::write("Unexpected parameter " + params[i] + " for command " + command_name
                           + "\n");
            valid = false;
        }
    }
    if (!valid)
    {
        error("Only " + cmds_to_string(allowed) + " are allowed\n");
    }
}
void check_for_duplicate_params(std::vector<std::string> const& params)
{
    std::unordered_set<std::string> found;
    for (auto i = 0U; i < params.size(); i += 2)
    {
        if (found.find(params[i]) == found.end())
        {
            found.insert(params[i]);
        }
        else
        {
            error("Duplicate parameter " + params[i] + "\n");
        }
    }
}

bool has_param(std::vector<std::string> const& params, std::string const& param)
{
    for (auto i = 0; i < params.size(); i += 2)
    {
        if (params[i] == param)
        {
            return true;
        }
    }
    return false;
}

std::string get_value(std::vector<std::string> const& params, std::string const& param)
{
    for (auto i = 0; i < params.size(); i += 2)
    {
        if (params[i] == param)
        {
            return params[i + 1];
        }
    }
    return "";
}

void error(std::string const& e)
{
    Console::write(e);
    throw Command_error();
}

void check_regex(std::string const& s)
{
    // check if regex can be created
    try
    {
        std::regex f(s);
    }
    catch (std::exception const& e)
    {
        error("Cannot create filter regex from given regex string " + s + "\n");
    }
}

} // namespace gtfsplanner
