#include "commands/mapcount.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/heatmap.h"

#include <algorithm>
#include <fstream>
#include <iomanip>

namespace gtfsplanner {
Mapcount_cmd::Mapcount_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}

void Mapcount_cmd::sanitize()
{
    std::vector<std::string> parameters {"mapmode", "heatmode"};
    auto const& params = get_parameters();
    if (!params.empty())
    {
        if (params.empty())
        {
            error("Missing parameter for departure command. Stop has to be given");
        }
        m_map = params[0];
        auto opt_params = std::vector<std::string>(params.begin() + 1, params.end());
        check_for_unknown_params(opt_params, parameters, "mapcount");
        check_for_duplicate_params(opt_params);

        retrieve_param(m_mapmode, opt_params, "mapmode");
        retrieve_param(m_heatmode, opt_params, "heatmode");
        check_mapmode(m_mapmode);
        check_heatmode(m_heatmode);
    }
}

void Mapcount_cmd::execute(gtfs::Dataset& dataset)
{
    std::unordered_map<std::pair<std::string, std::string>, size_t, pair_hash> count;

    // count all the connections between neighboring stops
    for (auto const& trip : dataset.trips)
    {
        for (auto i = 0U; i < trip.stop_time_indices.size() - 1; i++)
        {
            auto const& start = dataset.stop_times[trip.stop_time_indices[i]];
            auto const& end = dataset.stop_times[trip.stop_time_indices[i + 1]];
            auto start_id = start.stop_id;
            auto stop_id = end.stop_id;

            auto id = std::make_pair(start_id, stop_id);
            count[id]++;
        }
    }

    std::ofstream out(m_map);
    if (!out.is_open())
    {
        throw File_error("Could not open " + m_map + " for writing.");
    }

    Heatmap heatmap;
    std::unordered_map<std::string, Heatmap::Gradient> mapping = {
        {"rainbow", Heatmap::Gradient::Rainbow},
        {"monochrome", Heatmap::Gradient::Monochrome},
        {"viridis", Heatmap::Gradient::Viridis},
        {"inferno", Heatmap::Gradient::Inferno},
        {"wistia", Heatmap::Gradient::Wistia}};
    auto iter = mapping.find(m_heatmode);
    if (iter != mapping.end())
    {
        heatmap.set_gradient(iter->second);
    }
    // calculate the colors for the heatmap - take min and max values and then it's a linear interpolation with
    auto minmax =
        std::minmax_element(count.begin(), count.end(),
                            [](std::pair<std::pair<std::string, std::string>, size_t> const& lhs,
                               std::pair<std::pair<std::string, std::string>, size_t> const& rhs) {
                                return lhs.second < rhs.second;
                            });

    if (m_mapmode == "kml")
    {
        // write a kml file
        out << R"(<?xml version="1.0" encoding="UTF-8"?>)" << std::endl;
        out << R"(<kml xmlns="http://earth.google.com/kml/2.0">)" << std::endl;
        out << R"(<Document>)" << std::endl;
        // define the style - marker that can be colored, name will only be displayed when highlighting it on the map
        // to not clutter the map with labels
        out << R"(<Style id="defaultIcon">
	<LabelStyle>
		<scale>0</scale>
	</LabelStyle>
    <BalloonStyle>
      <text>$[description]</text>
    </BalloonStyle>
	<LineStyle><width>2</width>
	</LineStyle>
</Style>
<Style id="hoverIcon">
    <BalloonStyle>
      <text>$[description]</text>
    </BalloonStyle>
</Style>
<StyleMap id="defaultStyle">
	<Pair>
		<key>normal</key>
		<styleUrl>#defaultIcon</styleUrl>
	</Pair>
	<Pair>
		<key>highlight</key>
		<styleUrl>#hoverIcon</styleUrl>
	</Pair>
</StyleMap>)"
            << std::endl;

        // create the actual label entry, but sort them by alphabet
        struct Label
        {
            std::string name;
            double start_lon;
            double start_lat;
            double end_lon;
            double end_lat;
            Heatmap::Color color;
            double scaled_value;
            size_t total_value;
        };
        std::vector<Label> labels;
        for (auto const& entry : count)
        {
            auto start_iter = dataset.stop_id_to_idx.find(entry.first.first);
            check_iter(start_iter, dataset.stop_id_to_idx);
            auto const& start_station = dataset.stops[start_iter->second];

            auto end_iter = dataset.stop_id_to_idx.find(entry.first.second);
            check_iter(end_iter, dataset.stop_id_to_idx);
            auto const& end_station = dataset.stops[end_iter->second];

            // map value to range 0.0 - 1.0
            auto value = static_cast<double>(entry.second - minmax.first->second)
                         / static_cast<double>(minmax.second->second - minmax.first->second);
            auto color = heatmap.get_color(value);
            auto name = start_station.name + "-" + end_station.name;
            labels.push_back({name, start_station.lon, start_station.lat, end_station.lon,
                              end_station.lat, color, value, entry.second});
        }
        std::sort(labels.begin(), labels.end(),
                  [](Label const& lhs, Label const& rhs) { return lhs.name < rhs.name; });

        for (auto const& label : labels)
        {
            std::stringstream ss;
            ss << "ff";
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.b);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.g);
            ss << std::setw(2) << std::hex << std::setfill('0')
               << static_cast<unsigned>(label.color.r);

            out << "<Placemark><styleUrl>#defaultStyle</styleUrl>" << std::endl;
            out << "<Style><LineStyle><color>" << ss.str() << "</color></LineStyle></Style>"
                << std::endl;
            out << "<name>" << label.name << "</name>" << std::endl;
            out << "<LineString>" << std::endl;
            out << "<tesselate>1</tesselate>" << std::endl;
            out << "<coordinates>" << label.start_lon << "," << label.start_lat << ",0. "
                << label.end_lon << "," << label.end_lat << ",0."
                << "</coordinates>" << std::endl;
            out << "</LineString>" << std::endl;
            out << "<description>" << label.name << ": " << label.total_value << "</description>"
                << std::endl;
            out << "</Placemark>" << std::endl;
        }
        out << R"(</Document></kml>)";
    }
    else
    {
        //TODO implement
    }
}

void Mapcount_cmd::help()
{
    Console::write("Usage: mapcount <file> [mapmode <mode>] [heatmode <mode>]\n");
    Console::write("Writes a file marking the number of trips connecting neighbouring stations.\n");
    Console::write("mapmode (default: kml): Defines the output format. Only relevant when the map "
                   "option is used. Known formats: kml (for use in Google Earth), geojson (for use "
                   "in OpenStreetMap tools)\n");
    Console::write("heatmode (default: rainbow): Defines the color palette for the heatmap. Only "
                   "relevant when the map option is used. Known formats: inferno, monochrome "
                   "(black-white), rainbow, viridis, wistia\n");
}
} // namespace gtfsplanner
