#include "commands/list.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/exception.h"
#include "helpers/map_writer.h"
#include "helpers/progress_writer.h"

#include <algorithm>
#include <regex>

namespace gtfsplanner {
List_cmd::List_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}

void List_cmd::sanitize()
{
    std::vector<std::string> parameters {"filter", "map", "mapmode", "heatmode"};
    auto const& params = get_parameters();
    if (params.empty())
    {
        error("Missing parameter for list command. Type has to be given");
    }
    if (params[0] != "trips" && params[0] != "stations")
    {
        error(R"(Wrong type parameter for list command. Only "trips" and "stations" are allowed.)");
    }
    m_type = params[0];
    auto opt_params = std::vector<std::string>(params.begin() + 1, params.end());
    check_for_unknown_params(opt_params, parameters, "list");
    check_for_duplicate_params(opt_params);
    retrieve_param(m_filter, opt_params, "filter");
    retrieve_param(m_map, params, "map");
    retrieve_param(m_mapmode, params, "mapmode");
    retrieve_param(m_heatmode, params, "heatmode");

    if (has_param(opt_params, "map") && m_type != "stations")
    {
        error("Map option can only be used when listing stations.\n");
    }
    check_regex(m_filter);
    check_mapmode(m_mapmode);
    check_heatmode(m_heatmode);
}

void List_cmd::execute(gtfs::Dataset& dataset)
{
    std::vector<std::string> data;
    if (m_type == "trips")
    {
        data.reserve(dataset.trips.size());
        for (auto const& trip : dataset.trips)
        {
            data.push_back(build_route_name(dataset, trip));
        }
    }
    if (m_type == "stations")
    {
        data.reserve(dataset.stops.size());
        for (auto const& stop : dataset.stops)
        {
            data.push_back(stop.name);
        }
    }
    std::sort(data.begin(), data.end());
    std::regex filter(m_filter);
    size_t count = 0;
    for (auto const& entry : data)
    {
        if (m_filter.empty() || std::regex_match(entry, filter))
        {
            Console::write(entry + "\n");
            count++;
        }
    }
    Console::write(std::to_string(count) + " fitting entries found.\n");
    if (m_type == "stations" && !m_map.empty())
    {
        std::unordered_map<std::string, size_t> count;
        {
            Progress_writer progress("Preparing station heatmap", dataset.stops.size());
            for (auto const& stop : dataset.stops)
            {
                auto departures = get_departures(dataset, stop.id, Date {0, 0, 0}, Time {0, 0, 0},
                                                 Time {24, 0, 0});
                auto arrivals =
                    get_arrivals(dataset, stop.id, Date {0, 0, 0}, Time {0, 0, 0}, Time {24, 0, 0});
                if (arrivals.size() + departures.size() > 0)
                {
                    count.insert(std::make_pair(stop.id, departures.size() + arrivals.size()));
                }
                progress.update(1);
            }
        }
        write_heatmap(dataset, m_map, m_mapmode, m_heatmode, count);
    }
}


void List_cmd::help()
{
    Console::write(
        "Usage: list <type> [filter <regex>] [map <file>> [mapmode <mode>] [heatmode <mode>]\n");
    Console::write(
        "Shows a list of all items of the given type. The following types are supported:\n");
    Console::write(std::string("trips: Shows a list of all uniquely identifiable trips in the "
                               "current dataset, e.g. S1, RE")
                   + nbsp + "2, IC" + nbsp + "279, ICE" + nbsp + "459, R" + nbsp + "42, ...\n");
    Console::write("stations: Shows a list of all stations of the current dataset\n");
    Console::write("filter: Regular expression to reduce the amount of output. Note that if the "
                   "filter contains spaces, the whole string needs to be quoted like this: "
                   "\"filter with spaces\". \"list stations filter Frankfurt.*\" for example will "
                   "list all stations that begin with the word \"Frankfurt\".\n");
    Console::write("map: Writes a file marking the stations based on the number of trips that have "
                   "a stop there. Only relevant when the list type is \"stations\".\n");
    Console::write("mapmode (default: kml): Defines the output format. Only relevant when the map "
                   "option is used. Known formats: kml (for use in Google Earth), geojson (for use "
                   "in OpenStreetMap tools)\n");
    Console::write("heatmode (default: rainbow): Defines the color palette for the heatmap. Only "
                   "relevant when the map option is use.d Known formats: inferno, monochrome "
                   "(black-white), rainbow, viridis, wistia\n");
}
} // namespace gtfsplanner
