#include "commands/info.h"

#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/exception.h"

#include <algorithm>
#include <unordered_map>

namespace gtfsplanner {
Info_cmd::Info_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}

void Info_cmd::sanitize()
{
    auto const& params = get_parameters();
    if (!params.empty())
    {
        Console::write(
            "Unexpected parameters for info command. No additional parameters to be given.");
        throw Command_error();
    }
}

void show_stations_with_most_stops(gtfs::Dataset const& dataset)
{
    Console::write("Top 10 stops with most traffic (number of trips stopping)\n");
    using pair_type = std::pair<std::string, size_t>;
    std::unordered_map<std::string, size_t> count;
    for (auto const& stop : dataset.stop_times)
    {
        count[stop.stop_id]++;
    }
    for (auto i = 0U; i < 10; i++)
    {
        auto iter = std::max_element(
            count.begin(), count.end(),
            [](pair_type const& a, pair_type const& b) { return a.second < b.second; });
        auto idx_iter = dataset.stop_id_to_idx.find(iter->first);
        check_iter(idx_iter, dataset.stop_id_to_idx);
        auto const& stop = dataset.stops[idx_iter->second];
        Console::write(std::to_string(i + 1) + ". " + stop.name + ": "
                       + std::to_string(iter->second) + "\n");
        count.erase(iter);
    }
    Console::write("\n");
}

void show_trip_lengths(gtfs::Dataset const& dataset)
{
    using pair_type = std::pair<size_t, std::vector<std::string>>;
    std::unordered_map<size_t, std::vector<std::string>> trip_lengths_stations;
    std::unordered_map<size_t, std::vector<std::string>> trip_lengths_time;

    for (auto const& trip : dataset.trips)
    {
        auto const& stop_times = trip.stop_time_indices;
        trip_lengths_stations[stop_times.size()].push_back(trip.id);

        auto const& start = dataset.stop_times.at(stop_times.front());
        auto const& end = dataset.stop_times.at(stop_times.back());

        auto time = end.arrival_time - start.departure_time;
        trip_lengths_time[time.hour * 3600 + time.minute * 60 + time.second].push_back(trip.id);
    }

    auto iter =
        std::max_element(trip_lengths_stations.begin(), trip_lengths_stations.end(),
                         [](pair_type const& a, pair_type const& b) { return a.first < b.first; });
    Console::write("Trip(s) with most stops along the way: " + std::to_string(iter->first)
                   + " stations\n");
    for (auto const& trip_id : iter->second)
    {
        auto trip_data = build_trip_data(dataset, trip_id);
        print(trip_data, Trip_mode::DEPARTURE);
        Console::write("\n\n");
    }

    Console::write("Trip(s) with the longest duration: ");
    auto max_time_iter =
        std::max_element(trip_lengths_time.begin(), trip_lengths_time.end(),
                         [](pair_type const& a, pair_type const& b) { return a.first < b.first; });
    Console::write(to_string(Time {0, 0, static_cast<uint32_t>(max_time_iter->first)}) + "\n");
    for (auto const& trip_id : max_time_iter->second)
    {
        auto trip_data = build_trip_data(dataset, trip_id);
        print(trip_data, Trip_mode::DEPARTURE);
        Console::write("\n\n");
    }

    Console::write("Trip(s) with the shortest duration");
    auto min_time_iter =
        std::min_element(trip_lengths_time.begin(), trip_lengths_time.end(),
                         [](pair_type const& a, pair_type const& b) { return a.first < b.first; });
    Console::write(to_string(Time {0, 0, static_cast<uint32_t>(min_time_iter->first)}) + "\n");
    for (auto const& trip_id : min_time_iter->second)
    {
        auto trip_data = build_trip_data(dataset, trip_id);
        print(trip_data, Trip_mode::DEPARTURE);
        Console::write("\n\n");
    }
}

void Info_cmd::execute(gtfs::Dataset& dataset)
{
    Console::write("No. agencies:            " + std::to_string(dataset.agencies.size()) + "\n");
    Console::write("No. routes:              " + std::to_string(dataset.routes.size()) + "\n");
    Console::write("No. stops:               " + std::to_string(dataset.stops.size()) + "\n");
    Console::write("No. trips:               " + std::to_string(dataset.trips.size()) + "\n");
    Console::write("No. calendars:           " + std::to_string(dataset.calendars.size()) + "\n");
    Console::write("No. calendar exceptions: " + std::to_string(dataset.calendar_dates.size())
                   + "\n");
    Console::write("No. stop_times:          " + std::to_string(dataset.stop_times.size())
                   + "\n\n");

    show_stations_with_most_stops(dataset);

    show_trip_lengths(dataset);
}

void Info_cmd::help()
{
    Console::write("Usage: info\n");
    Console::write("Provides statistical information on the current dataset.\n");
}
} // namespace gtfsplanner
