#include "commands/departure.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/map_writer.h"
#include "helpers/string_functions.h"

namespace gtfsplanner {
Departure_cmd::Departure_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}


void Departure_cmd::sanitize()
{
    std::vector<std::string> parameters {"date", "start", "end", "map", "mapmode", "heatmode"};
    auto const& params = get_parameters();
    if (!params.empty())
    {
        if (params.empty())
        {
            error("Missing parameter for departure command. Stop has to be given");
        }
        m_stop = params[0];
        auto opt_params = std::vector<std::string>(params.begin() + 1, params.end());
        check_for_unknown_params(opt_params, parameters, "departure");
        check_for_duplicate_params(opt_params);
        retrieve_param(m_date, opt_params, "date");
        retrieve_param(m_start, opt_params, "start");
        retrieve_param(m_end, opt_params, "end");
        retrieve_param(m_map, opt_params, "map");
        retrieve_param(m_mapmode, opt_params, "mapmode");
        check_mapmode(m_mapmode);
        retrieve_param(m_heatmode, opt_params, "heatmode");
        check_heatmode(m_heatmode);
    }
}

void Departure_cmd::execute(gtfs::Dataset& dataset)
{
    auto station_iter = find_station(dataset, m_stop);
    if (station_iter != dataset.stops.end())
    {
        auto station_id = station_iter->id;
        auto trip_data = get_departures(dataset, station_id, m_date, m_start, m_end);

        for (auto& trip : trip_data)
        {
            Console::write(to_string(trip.stops[0].departure) + " " + trip.trip_type_name + " ");
            Console::write(to_upper(trip.stops[trip.stops.size() - 1].name) + "\n");

            trip.stops.erase(trip.stops.begin());

            Console::get().set_indent(9);
            print_trip_stops(trip, Trip_mode::DEPARTURE);
            print_trip_service_days(trip);
            Console::get().set_indent(0);
            Console::write("\n");
        }
        if (!m_map.empty())
        {
            std::unordered_map<std::string, size_t> count;
            // build map data to be displayed
            for (auto const& trip : trip_data)
            {
                for (auto const& stop : trip.stops)
                {
                    count[stop.stop_id]++;
                }
            }
            write_heatmap(dataset, m_map, m_mapmode, m_heatmode, count);
        }
    }
}

void Departure_cmd::help()
{
    Console::write("Usage: departure <stop> [date <YYYYMMDD>] [start <HH:MM::SS>] [end <HH:MM:SS>] "
                   "[map <file>] [mapmode <mode>] [heatmode <mode>]\n");
    Console::write(
        "Shows all departures from the given stop that fit the given optional criteria.\n");
    Console::write("date (default: none): only show departures for the given date.\n");
    Console::write("start (default: 00:00:00): only show trips departing equal to or later than "
                   "the given time.\n");
    Console::write("end (default: 24:00:00): only show trips departing earlier than or equal to "
                   "the given time.\n");
    Console::write("map: Writes a file marking the stations that can be reached as a heatmap based "
                   "on the number of departures to that location.\n");
    Console::write("mapmode (default: kml): Defines the output format. Only relevant when the map "
                   "option is used. Known formats: kml (for use in Google Earth), geojson (for use "
                   "in OpenStreetMap tools)\n");
    Console::write("heatmode (default: rainbow): Defines the color palette for the heatmap. Only "
                   "relevant when the map option is used. Known formats: inferno, monochrome "
                   "(black-white), rainbow, viridis, wistia\n");
}
} // namespace gtfsplanner
