#include "commands/count.h"

#include "commands/parser.h"
#include "gtfs_helpers.h"
#include "helpers/console.h"
#include "helpers/map_writer.h"
#include "helpers/string_functions.h"

namespace gtfsplanner {
Count_cmd::Count_cmd(std::vector<std::string> const& parameters) : Command(parameters) {}

void Count_cmd::sanitize()
{
    std::vector<std::string> parameters {"date", "start", "end",     "from",
                                         "to",   "map",   "mapmode", "heatmode"};
    auto const& params = get_parameters();
    if (!params.empty())
    {
        if (params.size() < 3)
        {
            error("Missing parameter(s) for count command. At least from and to stops have to be "
                  "given.");
        }
        check_for_unknown_params(params, parameters, "count");
        check_for_duplicate_params(params);
        if (has_param(params, "date"))
        {
            m_date = to<Date>(get_value(params, "date"));
        }
        if (has_param(params, "start"))
        {
            m_start = to<Time>(get_value(params, "start"));
        }
        if (has_param(params, "end"))
        {
            m_end = to<Time>(get_value(params, "end"));
        }
        if (has_param(params, "from"))
        {
            m_from = get_value(params, "from");
        }
        else
        {
            error("Missing from parameter for count command.");
        }
        if (has_param(params, "to"))
        {
            m_to = get_value(params, "to");
        }
        else
        {
            error("Missing to parameter for count command.");
        }
    }
}

void Count_cmd::execute(gtfs::Dataset& dataset)
{
    auto from_iter = find_station(dataset, m_from);
    auto to_iter = find_station(dataset, m_to);
    if (from_iter != dataset.stops.end() && to_iter != dataset.stops.end())
    {
        auto departures = get_departures(dataset, from_iter->id, m_date, m_start, m_end);
        auto to_id = to_iter->id;
        // filter out departures not going to m_to, i.e. to_id is not in the list of stops
        // also filter out if the to_id is there, but it's outside the given time frame
        departures.erase(std::remove_if(departures.begin(), departures.end(),
                                        [to_id, this](Trip_data const& trip) {
                                            auto iter =
                                                std::find_if(trip.stops.begin(), trip.stops.end(),
                                                             [to_id, this](Trip_stop const& stop) {
                                                                 return stop.stop_id == to_id;
                                                             });
                                            if (iter == trip.stops.end())
                                            {
                                                return true;
                                            }
                                            return (iter->arrival > m_end);
                                        }),
                         departures.end());

        // now departures contains all the connections we need
        Console::write("Found " + std::to_string(departures.size()) + " connections.\n");
        for (auto const& trip : departures)
        {
            print(trip, Trip_mode::DEPARTURE);
        }
    }
}

void Count_cmd::help()
{
    Console::write("Usage: count from <stop> to <stop> [start <time>} [end <time>]\n");
    Console::write("Counts the number of direct connections, i.e. no transfer between vehicles, "
                   "between the two given stops.\n");
    Console::write("from: Start stop of the route.\n");
    Console::write("to: End stop of the route.\n");
    Console::write("date (default: none): only show connections for the given date.\n");
    Console::write("start (default: 00:00:00): only show connections running equal to or later "
                   "than the given time.\n");
    Console::write("end (default: 24:00:00): only show connections running earlier than or equal "
                   "to the given time.\n");
    Console::write("\"count from \"Leipzig Hbf\" to \"Hamburg Hbf\" from 08:00:00 to 16:00:00\" "
                   "will show all direct connections between the two stations that depart after "
                   "08:00:00 and arrive until 16:00:00");
}
} // namespace gtfsplanner
