#ifndef GTFSPLANNER_TYPES_H
#define GTFSPLANNER_TYPES_H

#include "helpers/exception.h"

#include <cstdint>
#include <ctime>
#include <iostream>
#include <sstream>
#include <utility>

namespace gtfsplanner {

struct Time
{
    uint32_t hour;
    uint32_t minute;
    uint32_t second;

    void normalize()
    {
        while (second >= 60)
        {
            second -= 60;
            minute++;
        }
        while (minute >= 60)
        {
            minute -= 60;
            hour++;
        }
    }

    Time(uint32_t h, uint32_t m, uint32_t s) : hour(h), minute(m), second(s) { normalize(); }

    Time& operator+=(Time const& rhs)
    {
        hour += rhs.hour;
        minute += rhs.minute;
        second += rhs.second;

        normalize();
        return *this;
    }

    Time& operator-=(Time const& rhs)
    {
        // convert everything to seconds and use simple integer calculation
        int diff =
            (hour * 3600 + minute * 60 + second) - (rhs.hour * 3600 + rhs.minute * 60 + rhs.second);
        if (diff < 0)
        {
            throw Data_error("Time difference amounts to a negative time. This must never happen.");
        }
        hour = diff / 3600;
        diff -= hour * 3600;
        minute = diff / 60;
        diff -= minute * 60;
        second = diff;
        return *this;
    }
};

inline bool operator==(Time const& lhs, Time const& rhs)
{
    return lhs.hour == rhs.hour && lhs.minute == rhs.minute && lhs.second == rhs.second;
}

inline bool operator!=(Time const& lhs, Time const& rhs)
{
    return !(lhs == rhs);
}

inline bool operator<(Time const& lhs, Time const& rhs)
{
    if (lhs.hour != rhs.hour)
        return lhs.hour < rhs.hour;
    if (lhs.minute != rhs.minute)
        return lhs.minute < rhs.minute;
    return lhs.second < rhs.second;
}

inline bool operator>(Time const& lhs, Time const& rhs)
{
    return rhs < lhs;
}

inline bool operator<=(Time const& lhs, Time const& rhs)
{
    return lhs == rhs || lhs < rhs;
}

inline bool operator>=(Time const& lhs, Time const& rhs)
{
    return lhs == rhs || lhs > rhs;
}

inline Time operator+(Time const& lhs, Time const& rhs)
{
    Time result(lhs);
    result += rhs;
    return result;
}

inline Time operator-(Time const& lhs, Time const& rhs)
{
    Time result(lhs);
    result -= rhs;
    return result;
}

inline std::ostream& operator<<(std::ostream& stream, Time const time)
{
    auto hour = time.hour;
    while (hour > 23)
        hour -= 24;
    if (hour < 10)
        stream << "0";
    stream << hour << ":";
    if (time.minute < 10)
        stream << "0";
    stream << time.minute << ":";
    if (time.second < 10)
        stream << "0";
    stream << time.second;
    return stream;
}

inline std::string to_string(Time const& time)
{
    std::ostringstream ss;
    ss << time;
    return ss.str();
}

struct Date
{
    uint32_t year = 0;
    uint32_t month = 0;
    uint32_t day = 0;
};

/// \return day of week, Sunday == 0, Monday == 1, ...
inline size_t get_day_of_week(Date const& date)
{
    // calculate day of week for given date
    // algorithm according to ahttps://en.wikipedia.org/wiki/Determination_of_the_day_of_the_week#Sakamoto's_methods
    int t[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};
    int y = date.year;
    int m = date.month;
    int d = date.day;
    y -= m < 3;
    auto dayofweek = (y + y / 4 - y / 100 + y / 400 + t[m - 1] + d) % 7;
    return dayofweek;
}

inline bool operator==(Date const& lhs, Date const& rhs)
{
    return lhs.year == rhs.year && lhs.month == rhs.month && lhs.day == rhs.day;
}
inline bool operator!=(Date const& lhs, Date const& rhs)
{
    return !(lhs == rhs);
}

inline bool operator<(Date const& lhs, Date const& rhs)
{
    if (lhs.year != rhs.year)
        return lhs.year < rhs.year;
    if (lhs.month != rhs.month)
        return lhs.month < rhs.month;
    return lhs.day < rhs.day;
}

inline bool operator>(Date const& lhs, Date const& rhs)
{
    return rhs < lhs;
}

inline bool operator<=(Date const& lhs, Date const& rhs)
{
    return lhs == rhs || lhs < rhs;
}

inline bool operator>=(Date const& lhs, Date const& rhs)
{
    return lhs == rhs || lhs > rhs;
}

inline Date& operator++(Date& a)
{
    // have to do weird conversions with date/time formats as C++ does not really provide a useful date library
    struct tm date = {0, 0, 12};
    date.tm_year = a.year - 1900;
    date.tm_mon = a.month - 1;
    date.tm_mday = a.day;
    const time_t day = 24 * 3600; // num_seconds per day
    time_t date_seconds = mktime(&date) + day;
    date = *localtime(&date_seconds);
    a.year = date.tm_year + 1900;
    a.month = date.tm_mon + 1;
    a.day = date.tm_mday;
    return a;
}

inline Date operator++(Date& a, int)
{
    Date result(a);
    ++a;
    return result;
}

inline std::ostream& operator<<(std::ostream& stream, Date const& date)
{
    if (date.day < 10)
        stream << "0";
    stream << date.day << ".";
    if (date.month < 10)
        stream << "0";
    stream << date.month << ".";
    stream << date.year;
    return stream;
}

// needed because standard doesn't provide one
struct pair_hash
{
    template <typename T1, typename T2>
    std::size_t operator()(const std::pair<T1, T2>& pair) const
    {
        return std::hash<T1>()(pair.first) ^ std::hash<T2>()(pair.second);
    }
};

} // namespace gtfsplanner

#endif
