#ifndef GTFSPLANNER_STRING_FUNCTIONS_H
#define GTFSPLANNER_STRING_FUNCTIONS_H

#include "types.h"

#include <regex>
#include <sstream>
#include <string>
#include <vector>

namespace gtfsplanner {

constexpr const char* nbsp = "\xC2\xA0";

// replace tabs with spaces, trim excessive whitespaces
std::string sanitize_whitespaces(std::string const& input);

// splits the given string at spaces, unless it is quoted
// Quotes inside quotes are to be handled by escaping them using double quotes
// i.e. if input = 'This is a string' (without the quotes), the result
// will be [This,is,a,string]. If the input is 'This is "a string ""with"" quotes"', the
// result will be [This,is,a string "with" quotes]
std::vector<std::string> tokenize_with_quotes(std::string const& input, char delimiter);

std::vector<std::string> tokenize(std::string const& input, char delimiter);

template <typename T>
T to(std::string const& s)
{
    std::istringstream ss(s);
    T result;
    ss >> result;
    if (ss.fail())
    {
        throw std::invalid_argument("could not convert input " + s);
    }
    return result;
}

template <>
inline Time to<Time>(std::string const& s)
{
    uint32_t hms[3] = {0, 0, 0};
    size_t index = 0;
    for (auto const& c : s)
    {
        if (c == ':')
        {
            index++;
            if (index > 2)
            {
                break;
            }
        }
        else if (isdigit(c))
        {
            hms[index] = hms[index] * 10 + c - '0';
        }
        else
        {
            // invalid character -> trigger error condition
            index = 0;
            break;
        }
    }
    if (index != 2)
    {
        throw std::runtime_error("Cannot parse " + s + " as time.");
    }

    return Time {hms[0], hms[1], hms[2]};
}

template <>
inline Date to<Date>(std::string const& s)
{
    std::regex date("([0-9]{4})([0-9]{2})([0-9]{2})");
    std::smatch match;
    if (!std::regex_match(s, match, date))
    {
        throw std::runtime_error("Cannot parse " + s + " as date.");
    }

    return Date {to<uint32_t>(match[1].str()), to<uint32_t>(match[2].str()),
                 to<uint32_t>(match[3].str())};
}


template <typename T>
T to(char c)
{
    if (c < '0' || c > '9')
    {
        std::string s(1, c);
        throw std::invalid_argument("could not convert input " + s);
    }
    return static_cast<T>(c - '0');
}

size_t fuzzy_string_compare(std::string const& a, std::string const& b);

std::string to_upper(std::string const& input);

size_t length_utf8(std::string const& s);
} // namespace gtfsplanner

#endif
