#ifndef GTFSPLANNER_GTFSTYPES_H
#define GTFSPLANNER_GTFSTYPES_H

#include "types.h"

#include <unordered_map>
#include <vector>

namespace gtfsplanner {
namespace gtfs {

struct Agency
{
    std::string id;
    std::string name;
    std::string url;
    std::string timezone;
};

struct Stop
{
    std::string id;
    std::string name;
    double lat;
    double lon;
};

enum class Route_type
{
    TRAM = 0,
    SUBWAY = 1,
    TRAIN = 2,
    BUS = 3,
    FERRY = 4,
    CABLECAR = 5,        // San Francisco
    CABLE_TRANSPORT = 6, // e.g. in the mountains
    FUNICULAR = 7,       // e.g. Dresden
    TROLLEY_BUS = 11,
    MONORAIL = 12 // e.g. Wuppertal
};

struct Route
{
    std::string id;
    std::string agency_id;
    std::string short_name;
    std::string long_name;
    Route_type type;
};

enum class Direction
{
    IN_DIRECTION = 0,
    IN_REVERSE_DIRECTION = 1
};

struct Trip
{
    std::string route_id;
    std::string service_id;
    std::string id;
    Direction direction_id;

    // additional indexing info, not part of actual gtfs format
    std::vector<size_t>
        stop_time_indices; // all stops of the trip, in order, indexes into stop_times array of the dataset
};

enum class Access
{
    YES = 0,
    NO = 1,
    ONLY_WITH_TELEPHONIC_APPROVAL = 2,
    ONLY_WITH_DRIVER_APPROVAL = 3
};

struct Stop_time
{
    std::string trip_id;
    Time arrival_time;
    Time departure_time;
    std::string stop_id;
    uint32_t stop_sequence;
    Access pickup_type;
    Access drop_off_type;
};

struct Calendar
{
    std::string service_id;
    bool
        monday; // true: service is scheduled for all Mondays in given time range, false: service is not operating on Mondays
    bool tuesday; // see monday
    bool wednesday;
    bool thursday;
    bool friday;
    bool saturday;
    bool sunday;
    Date start_date;
    Date end_date;
};

enum class Service_exception
{
    ADDED = 1,
    REMOVED = 2
};

struct Calendar_date
{
    std::string service_id;
    Date date;
    Service_exception exception_type;
};

enum class Transfer_type
{
    TRANSFER_PROPOSED = 0,
    WAITING = 1,
    TRANSFER_TIME_NEEDED = 2,
    NO_TRANSFER_POSSIBLE = 3
};

struct Transfer
{
    std::string from_stop_id;
    std::string to_stop_id;
    Transfer_type transfer_type;
    uint32_t min_transfer_time_sec;
};

struct Dataset
{
    std::vector<Agency> agencies;
    std::vector<Stop> stops;
    std::vector<Route> routes;
    std::vector<Trip> trips;
    std::vector<Stop_time> stop_times;
    std::vector<Calendar> calendars;
    std::vector<Calendar_date> calendar_dates;
    std::vector<Transfer> transfers;

    // additional indexing information
    std::unordered_map<std::string, size_t> route_id_to_idx;
    std::unordered_map<std::string, size_t> stop_id_to_idx;
    std::unordered_map<std::string, size_t> trip_id_to_idx;
};
} // namespace gtfs

// check the validity of iterators
template <typename T, typename U>
void check_iter(T iterator, U const& data)
{
    if (iterator == data.end())
    {
        throw Data_error("Inconsistent data detected, invalid iterator found");
    }
}
} // namespace gtfsplanner

#endif
