#ifndef GTFSPLANNER_COMMANDPARSER_H
#define GTFSPLANNER_COMMANDPARSER_H

#include "commands/command.h"
#include "helpers/string_functions.h"

#include <memory>
#include <string>
#include <vector>

namespace gtfsplanner {
/// parse the commandline parameters and build a command list
/// \param args the arguments to be parsed
/// \return list of commands to be executed
/// \throws Exception if args cannot be parsed
std::vector<std::unique_ptr<Command>> parse(std::vector<std::string> const& args);

/// parse the given input into a command
/// \param input the string to be parsed
/// \return a command
/// \throw Exception if input cannot be turned into a command
std::unique_ptr<Command> parse_command(std::string const& input);

/// builder function to create commands
/// \param cmd the command string without parameters (e.g. "load", "route", ...)
/// \param paramters the parameters to be given to the command
/// \return a command, might be null if there is no fitting command
std::unique_ptr<Command> build_command(std::string const& cmd,
                                       std::vector<std::string> const& parameters);

std::string cmds_to_string(std::vector<std::string> const& commands);

/// checks the given params for unknown commands or parameters
/// This exploits the fact that the parameters by convention follow a name value scheme, so every other
/// value is a parameter.
/// \param params the parameters to be checked
/// \param allowed the parameters to look for
/// \param command_name the name of the current command
void check_for_unknown_params(std::vector<std::string> const& params,
                              std::vector<std::string> const& allowed,
                              std::string const& command_name);
void check_for_duplicate_params(std::vector<std::string> const& params);

bool has_param(std::vector<std::string> const& params, std::string const& param);
std::string get_value(std::vector<std::string> const& params, std::string const& param);

void error(std::string const& e);

template <typename T>
void retrieve_param(T& result, std::vector<std::string> const& params, std::string const& key)
{
    if (has_param(params, key))
    {
        result = to<T>(get_value(params, key));
    }
}
template <>
inline void retrieve_param(std::string& result,
                           std::vector<std::string> const& params,
                           std::string const& key)
{
    if (has_param(params, key))
    {
        result = get_value(params, key);
    }
}

inline void check_mapmode(std::string const& mode)
{
    if (mode != "kml" && mode != "geojson")
    {
        error(R"(Mapmode must be either "kml" or "geojson".)");
    }
}

inline void check_heatmode(std::string const& mode)
{
    if (mode != "rainbow" && mode != "monochrome" && mode != "inferno" && mode != "viridis"
        && mode != "wistia")
    {
        error(R"(Heatmode must be "inferno", "monochrome", "rainbow", "viridis" or "wistia".)");
    }
}

void check_regex(std::string const& s);

} // namespace gtfsplanner

#endif
